#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/i2c.h>
#include <linux/input.h>
#include <linux/input-polldev.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/pm.h>
#include <linux/pm_runtime.h>
#include <linux/string.h>
#include <linux/gpio.h>
#include <linux/regulator/consumer.h>
#include <linux/firmware.h>
#include <linux/wakelock.h>
#include <linux/of_gpio.h>
#ifdef CONFIG_FB
#include <linux/notifier.h>
#include <linux/fb.h>
#endif

#include "CwMcuSensor.h"
#include "lpc5400x_update.h"

static struct wake_lock fwupdate_wake_lock;
static struct wake_lock wakeup_event_wake_lock;

/* -- FEATURE FLAGS SECTION START -- */
/* turn on to support dvt1 prototype */
#define CONFIG_SNR_HUB_DVT1_COMPATIBLE
/* turn on interrupt reporting if defined */
#define CWMCU_INTERRUPT
/* turn on to recover after detected continous i2c errors to avoid panic. */
//#define CONFIG_SNR_HUB_I2C_HACK
/* turn on to avoid compass current leak align with fw before v08.25 */
//#define CONFIG_SNR_HUB_MAG_LEAK_HACK
/* TODO: Move to makefile to auto mark per build type */
//#define CONFIG_SNS_HUB_DEBUG
//#define CWMCU_POLLING_DEVICE
/* -- FEATURE FLAGS SECTION END -- */

#define SUSPEND_ENABLE_LIST     ((1<<CW_SINGLE_SNAP) | (1<<CW_CONTEXT_AWARE) )

#define SP_ENABLE_LIST     ((1<<CW_SINGLE_SNAP) | (1<<CW_CONTEXT_AWARE) | (1<<CW_REAR_CAMERA) | (1<<CW_PROXIMITY_GESTURE) | (1<<CW_PROXIMITY) )
#define IMMEDIATE_SENSORS_LIST ( (1<<CW_PROXIMITY) | (1<<CW_SINGLE_SNAP) | (1<<CW_REAR_CAMERA) )

#define TAG_PROX                 "PROX"
#define TAG_CONTAWARENESS        "CONTEXT-AWARENESS"
#define TAG_VOICE_TRIGGER        "VOICE"
#define TAG_SCREENON             "SCREENON"
#define TAG_GESTURE              "PROX-GESTURE"
#define TAG_SSNAP                "S-SNAP"
#define print_event(id, data) do { printk("CWMCU Event %s : %d \n", id,data); } while (0)


#define CWMCU_MEM_SENSORS_LIST  ( (1<<CW_ACCELERATION) | (1<<CW_MAGNETIC) | (1<<CW_GYRO) | (1<<CW_PRESSURE) \
									| (1<<CW_ORIENTATION) | (1<<CW_ROTATIONVECTOR) | (1<<CW_LINEARACCELERATION) | (1<<CW_GRAVITY) \
									| (1<<CW_MAGNETIC_UNCALIBRATED) | (1<<CW_GYROSCOPE_UNCALIBRATED) | (1<<CW_GAME_ROTATION_VECTOR) | (1<<CW_GEOMAGNETIC_ROTATION_VECTOR) )

									
#define VERBOSE(format,...) do {            \
                                if (flagVerbose) printk(KERN_DEBUG format, ##__VA_ARGS__); \
                            } while (0)


/* GPIO for MCU control */
//#define GPIO_CW_MCU_WAKE_UP		67
//#define GPIO_CW_MCU_BOOT		139
//#define GPIO_CW_MCU_RESET		140
//#define GPIO_CW_MCU_INTERRUPT	73


#define DPS_MAX			        (1 << (16 - 1))


/* Input poll interval in milliseconds */
#define CWMCU_POLL_INTERVAL	10 
//#define CWMCU_POLL_MAX		200
#define CWMCU_POLL_MIN		10

#define FT_VTG_MIN_UV		2600000
#define FT_VTG_MAX_UV		3300000
#define FT_I2C_VTG_MIN_UV	1800000
#define FT_I2C_VTG_MAX_UV	1800000


#define I2C_ERRS_MAX        10

static DEFINE_MUTEX(cwmcu_lock);

// Flag to enable/disable verbose message, for debug convinience.
static volatile int flagVerbose;

struct CWMCU_data {
	struct i2c_client *client;
	struct regulator *vdd;
	struct regulator *vcc_i2c;
	struct input_dev *input;
	struct timeval previous;
	struct timeval now;
	int mcu_mode;
	int mcu_status;
	int mcu_power;
    int mcu_screen;
    int mcu_reset;


	struct workqueue_struct *driver_wq;
	struct work_struct work;

	/* enable & batch list */
	uint32_t enabled_list[CW_HANDLE_ID_END];
	uint32_t batched_list[CW_HANDLE_ID_END];
	uint32_t batch_timeout[CW_HANDLE_ID_END];
	uint32_t flush_list[CW_HANDLE_ID_END];
	
	/* report time */
	uint32_t	sensor_timeout[CW_HANDLE_ID_END][CW_SENSORS_ID_END];
    uint8_t		report_period[CW_HANDLE_ID_END][CW_SENSORS_ID_END];/* ms */
	int64_t		pre_sensors_time[CW_HANDLE_ID_END][CW_SENSORS_ID_END];/* pre_timestamp(us) */
	int64_t		time_diff[CW_HANDLE_ID_END][CW_SENSORS_ID_END];

	uint32_t suspend_enabled_list;
    uint32_t step_counter;
    

    uint32_t interrupt_status;

	
    uint32_t    update_list[CW_HANDLE_ID_END];
	uint32_t    sensors_report_list;

	/* power status */
	int power_on_list;

	/* debug */
	int debug_count;

	/* calibrator */
	uint8_t calib_cmd;
	uint8_t calib_id;

	/* selftest */
	uint8_t selftest_id;

    /* gpio */
    int irq_gpio;
    int mic_gpio;
    int wakeup_gpio;
    int reset_gpio;
    int reset_gpio_dvt1;
    int compass_reset_gpio;

	int cmd;
	int addr;
	int len;
	int value;
	int mcu_slave_addr;
	int fwupdate_status;
	int cw_i2c_rw;	/* r = 0 , w = 1 */
	int cw_i2c_len;
	uint8_t cw_i2c_data[300];
	char fw_name[32];
	int sensorhub_data_mode;

	/* check mcu sensor */
    //uint8_t check_muc_sensor_alive_flag;
    //int poll_count_num;
    //int preContinuousSensor[3][3];
    //int curContinuousSensor[3][3];
    //int sensor_error_count[3];
    //int mcu_reset_error_count;
    int pre_reset_count;

#ifdef CONFIG_SNS_HUB_DEBUG
	bool power_test_on;
#endif
	/* Assure not to access i2c if mcu still interrupt after suspend due to unknown intenal issue. */
    bool is_suspend;

	uint32_t preCTimestame;
	uint32_t preTimestameAcc;
	uint32_t preCTSTimestame[3];

	uint32_t uncalibratedGyro[CW_HANDLE_ID_END][3];
	uint32_t uncalibratedMag[CW_HANDLE_ID_END][3];
	

};

struct CWMCU_data *sensor;

static void get_mag_vendor(void);

/* normalmode: bool: If true, then mcu will reboot to normal mode, otherwise reboot to second-bootloader mode */
static void CWMCU_reset_mcu(bool normalmode)
{
    if (normalmode) {
        gpio_direction_output(sensor->irq_gpio, 1);
        gpio_direction_output(sensor->mic_gpio, 1);
    } else {
        gpio_direction_output(sensor->irq_gpio, 0);
        gpio_direction_output(sensor->mic_gpio, 0);
    }

    sensor->pre_reset_count = sensor->mcu_reset;

    gpio_direction_output(sensor->reset_gpio, 1);
#ifdef CONFIG_SNR_HUB_DVT1_COMPATIBLE
    gpio_direction_output(sensor->reset_gpio_dvt1, 1);
#endif
    usleep_range(19000, 20000);
    gpio_set_value(sensor->reset_gpio, 0);
#ifdef CONFIG_SNR_HUB_DVT1_COMPATIBLE
    gpio_set_value(sensor->reset_gpio_dvt1, 0);
#endif
    usleep_range(19000, 20000);
    gpio_set_value(sensor->reset_gpio, 1);
#ifdef CONFIG_SNR_HUB_DVT1_COMPATIBLE
    gpio_set_value(sensor->reset_gpio_dvt1, 1);
#endif

    usleep(10000);
    gpio_direction_input(sensor->irq_gpio);
    gpio_direction_input(sensor->mic_gpio);
}

/* helper function to avoid vulnerable qualcomm i2c timeout oops.
 * Each i2c ops status should be checked by this function.
 */
static void check_i2c_error_cnt(bool is_err)
{
#ifdef CONFIG_SNR_HUB_I2C_HACK
    static int i2c_err_num = 0;

    if (is_err) {
        i2c_err_num++;
        if (i2c_err_num > I2C_ERRS_MAX) {
            printk("CWMCU too much i2c err, reset mcu!!\n");
            CWMCU_reset_mcu(true);
            i2c_err_num = 0;
        }
    } else
        i2c_err_num = 0;
#endif
}

static int CWMCU_i2c_write(struct CWMCU_data *sensor, u8 reg_addr, u8 *data, u8 len)
{
    int dummy;
    int i;

	mutex_lock(&cwmcu_lock);
    for (i = 0; i < len; i++) {
        dummy = i2c_smbus_write_byte_data(sensor->client, reg_addr++, data[i]);
        check_i2c_error_cnt((dummy < 0));
        if (dummy < 0) {
            pr_err("CWMCU i2c write error =%d\n", dummy);
			mutex_unlock(&cwmcu_lock);
            return dummy;
        }
    }
	mutex_unlock(&cwmcu_lock);
    return 0;
}

/* Returns the number of read bytes on success */
static int CWMCU_i2c_read(struct CWMCU_data *sensor, u8 reg_addr, u8 *data, u8 len)
{
    int dummy;

	mutex_lock(&cwmcu_lock);
    dummy = i2c_smbus_read_i2c_block_data(sensor->client, reg_addr, len, data);
    if (len)
        check_i2c_error_cnt((dummy == 0));

	mutex_unlock(&cwmcu_lock);
    return dummy;
}

/* write format    1.slave address  2.data[0]  3.data[1] 4.data[2] */
static int CWMCU_write_i2c_block(struct CWMCU_data *sensor, u8 reg_addr, const u8 *data, u8 len)
{
    int dummy;

	mutex_lock(&cwmcu_lock);
    dummy = i2c_smbus_write_i2c_block_data(sensor->client, reg_addr, len, data);
    check_i2c_error_cnt((dummy < 0));
    if (dummy < 0) {
        pr_err("CWMCU i2c write error =%d\n", dummy);
		mutex_unlock(&cwmcu_lock);
        return dummy;
    }
	mutex_unlock(&cwmcu_lock);
    return 0;
}

static int CWMCU_write_i2c_multiple(struct CWMCU_data *sensor, u8 reg_addr, const u8 *data, int len)
{
    int dummy;
    unsigned char buffer[len+1];

	mutex_lock(&cwmcu_lock);

    buffer[0] = reg_addr;

    memcpy(&buffer[1],data,len);

    dummy = i2c_master_send(sensor->client, buffer, len+1);
    check_i2c_error_cnt((dummy < 0));
    if (dummy < 0) {
        printk("CWMCU i2c master send fail! error=%d\n", dummy);
		mutex_unlock(&cwmcu_lock);
        return dummy;
    }
	mutex_unlock(&cwmcu_lock);
    return 0;
}

static int CWMCU_read_i2c_multiple(struct CWMCU_data *sensor, u8 reg_addr, u8 *data, int len)
{
    int dummy;

	mutex_lock(&cwmcu_lock);
	dummy = i2c_master_send(sensor->client, &reg_addr, 1);
    if (dummy < 0) {
        printk("CWMCU i2c master send fail! error=%d\n", dummy);
		mutex_unlock(&cwmcu_lock);
        return dummy;
    }
	
	dummy = i2c_master_recv(sensor->client, data, len);
	if (dummy < 0) {
		printk("CWMCU i2c master recv fail! error=%d\n", dummy);
		mutex_unlock(&cwmcu_lock);
		return dummy;
	}
	mutex_unlock(&cwmcu_lock);
	return 0;
}


/*
static int CWMCU_write_serial(u8 *data, int len)
{
	int dummy;
	dummy = i2c_master_send(sensor->client, data, len);
	if (dummy < 0) {
		pr_err("i2c write error =%d\n", dummy);
		return dummy;
	}
	return 0;
}

static int CWMCU_read_serial(u8 *data, int len)
{
	int dummy;
	dummy = i2c_master_recv(sensor->client, data, len);
	if (dummy < 0) {
		pr_err("i2c read error =%d\n", dummy);
		return dummy;
	}
	return 0;
}
*/


#if 0
static void cwmcu_powermode_switch(SWITCH_POWER_ID id, int onoff)
{
	if (onoff) {
		if (sensor->power_on_list == 0) {
			gpio_set_value(GPIO_CW_MCU_WAKE_UP, onoff);
		}
		sensor->power_on_list |= ((uint32_t)(1) << id);
		if (id != SWITCH_POWER_INTERRUPT) {
			printk("--CWMCU-- power mode sleep~!!!\n");
			usleep_range(500, 1000);
		}
	} else {
		sensor->power_on_list &= ~(1 << id);
		if (sensor->power_on_list == 0) {
			gpio_set_value(GPIO_CW_MCU_WAKE_UP, onoff);
		}
	}
	/* printk(KERN_DEBUG "--CWMCU--%s id = %d, onoff = %d\n", __func__, id, onoff); */
}
#endif

void print_sp_event(uint8_t id, int data) {
	char buf[32];
	switch (id) {
	case CW_SINGLE_SNAP:
		sprintf(buf, "%s",TAG_SSNAP);
		break;
	case CW_CONTEXT_AWARE:
		sprintf(buf, "%s",TAG_CONTAWARENESS);
		break;
	case CW_REAR_CAMERA:
		sprintf(buf, "%s",TAG_SCREENON);
		break;
	case CW_PROXIMITY_GESTURE:
		sprintf(buf, "%s",TAG_GESTURE);
		break;
	case CW_PROXIMITY:
		sprintf(buf, "%s",TAG_PROX);
		break;
	}

	print_event(buf, data);
}

static int cwmcu_get_version(void)
{
    int ret;
    uint8_t count;
    uint8_t data[6] = {0};

    printk("--CWMCU-- %s\n", __func__);

    for (count = 0; count < 3; count++) {
        //if ((ret = CWMCU_i2c_read(sensor, CW_FWVERSION, data, 2)) >= 0) {
		if ((ret = CWMCU_read_i2c_multiple(sensor, CW_FWVERSION, data, 2)) >= 0) {
            printk("--CWMCU-- CHECK_FIRMWAVE_VERSION : %d.%d\n", data[0],data[1]);
            return 0;
        }
        mdelay(20);
    }
    
    printk("--CWMCU-- Cannot get fw version, [ret:%d]\n", ret);
    return ret;
}


static void cwmcu_restore_status(void)
{
    int id,hd;
#ifdef CONFIG_SNS_HUB_DEBUG
    u8 list;
#endif
	uint32_t enable_list_temp;
    uint8_t activate[2] = {0,};
	uint8_t batch[7] = {0,};
	uint8_t pedometer[4] = {0,};
	uint8_t mSensorId, mDelayMs;
	uint32_t mTimeout = 0;

    printk("--CWMCU-- %s\n", __func__);

#ifdef CONFIG_SNS_HUB_DEBUG
    list  = (sensor->power_test_on)? 0: 1;
    printk("CWMCU write CW_SET_INTERRUPT_ENABLE, %d\n", list);
    if (CWMCU_write_i2c_block(sensor, CW_SET_INTERRUPT_ENABLE, &list, 1) < 0) {
        printk("CWMCU Write CW_SET_INTERRUPT_ENABLE error.\n");
    }
#endif

    //Host sensor enable & set delay
	for(hd=0;hd<CW_HANDLE_ID_END;hd++) {
		enable_list_temp = sensor->enabled_list[hd];
		for (id = 0; id < CW_SENSORS_ID_END; id++) {
			if (enable_list_temp & (1<<id)) {
				//CWMCU_write_i2c_block(sensor, CW_HOST_ACTIVATE, &list, 1);
				mSensorId = (uint8_t)(id + (hd*32));
				mDelayMs  = (uint8_t)sensor->report_period[hd][id];
				mTimeout  = (uint32_t)sensor->sensor_timeout[hd][id];
				
				/* enable sensor */
				activate[0] = mSensorId;
				activate[1] = 1;
				if(CWMCU_write_i2c_block(sensor, CW_HOST_ACTIVATE, activate, 2) < 0) {
					printk( "--CWMCU-- active sensor error. [i2c]\n");
				}
				
				/* set sensor delay */
				//id, flag, minDelay, timeout
				batch[0] = mSensorId;
				batch[1] = (uint8_t)hd;
				batch[2] = mDelayMs;
				memcpy(batch+3,&mTimeout,sizeof(uint32_t));
				if(CWMCU_write_i2c_block(sensor, CW_HOST_BATCH, batch, 7) < 0) {
					printk("--CWMCU-- batch set fail. [i2c]\n");
				}
			}
		}
	}


    /* pedometer */
    memcpy(pedometer,&sensor->step_counter,sizeof(uint8_t)*4);
    if(CWMCU_write_i2c_multiple(sensor, CW_STEP_COUNTER_SET, pedometer, 4)<0) {
        printk("--CWMCU-- restore pedometer error!!\n");
    }
}


static void cwmcu_send_time_sync_event(void)
{
	printk( "--CWMCU-- send time sync event!\n");
	input_report_abs(sensor->input, CW_ABS_TIMESYNC, 0);
	input_sync(sensor->input);

	input_report_abs(sensor->input, CW_ABS_TIMESYNC, 1);
	input_sync(sensor->input);
}

static void cwmcu_send_flush(int id)
{
	uint32_t flush_data = 0;

	flush_data = ((u32)CW_META_DATA << 16) | id;
	printk("--CWMCU-- flush sensor: %d auto return~!!\n", id);
	input_report_abs(sensor->input, CW_ABS_Z, flush_data);
	input_sync(sensor->input);
	/* reset flush event for ABS_Z*/
	input_report_abs(sensor->input, CW_ABS_Z, 0xFF0000);
	input_sync(sensor->input);
}


static int cwmcu_stream_mode_read(struct CWMCU_data *sensor, int wakeup)
{
    uint8_t data[12] = {0,};
    uint32_t data_event[9]={0,};
    int i = 0;
	int id_check=0;
	int stream_q_count = 0, onchange_q_count = 0;
	uint8_t stream_mode_reg_count, stream_mode_reg_event;
	uint8_t stream_mode_onchange_reg_count, stream_mode_onchange_reg_event;

	uint32_t rawdata[6] = {0,};
	uint32_t timestamp = 0;
	int spdata = 0;
	uint8_t sensor_id = 0;

    if (sensor->mcu_mode == CW_BOOT || sensor->mcu_power == CW_BOOT) {
        return 0;
    }


	if(wakeup==0) {
		stream_mode_reg_count = CW_STREAM_NONWAKEUP_COUNT;
		stream_mode_reg_event = CW_STREAM_NONWAKEUP_EVENT;
		stream_mode_onchange_reg_count = CW_STREAM_NONWAKEUP_ONCHANGEQ_COUNT;
		stream_mode_onchange_reg_event = CW_STREAM_NONWAKEUP_ONCHANGEQ_EVENT;
	}
	else if(wakeup==1) {
		stream_mode_reg_count = CW_STREAM_WAKEUP_COUNT;
		stream_mode_reg_event = CW_STREAM_WAKEUP_EVENT;
		stream_mode_onchange_reg_count = CW_STREAM_WAKEUP_ONCHANGEQ_COUNT;
		stream_mode_onchange_reg_event = CW_STREAM_WAKEUP_ONCHANGEQ_EVENT;
	}
	else {
		printk("--CWMCU-- handle[%d]: parameter wakeup is error value.\n", wakeup);
		return 0;
	}



	if (CWMCU_i2c_read(sensor, stream_mode_reg_count, data, 1) >= 0) {
		stream_q_count = data[0];
	}
	else {
		stream_q_count = 0;
		printk("--CWMCU-- handle[%d]: read stream event count failed[i2c]!!\n", wakeup);
	}
	VERBOSE("--CWMCU-- handle[%d]: stream queue count:%d \n", wakeup, stream_q_count);


	for (i = 0; i < stream_q_count; i++) {
		/* read 11byte */
		if (CWMCU_i2c_read(sensor, stream_mode_reg_event, data, 11) >= 0) {
			//VERBOSE("--CWMCU-- handle[%d]: read stream data = %d/%d\n", wakeup, i,stream_q_count);

			if (data[0] != CWMCU_NODATA) {
				sensor_id = data[0] % 32;
				if(sensor_id & IMMEDIATE_SENSORS_LIST) {
					id_check = data[0];
				}
				else {
					id_check = data[0] + 32 * wakeup;
				}

				data_event[0] = ((u32)id_check << 16) | (((u32)data[2] << 8) | (u32)data[1]);
				data_event[1] = ((u32)id_check << 16) | (((u32)data[4] << 8) | (u32)data[3]);
				data_event[2] = ((u32)id_check << 16) | (((u32)data[6] << 8) | (u32)data[5]);
				data_event[3] = (uint32_t)(data[10]<<24) | (uint32_t)(data[9]<<16) | (uint32_t)(data[8]<<8) | (uint32_t)data[7];

				rawdata[0] = (((u32)data[2]  << 8) | (u32)data[1]);
                rawdata[1] = (((u32)data[4]  << 8) | (u32)data[3]);
                rawdata[2] = (((u32)data[6]  << 8) | (u32)data[5]);
				timestamp = data_event[3];
                
				if(data[0] == CW_ORIENTATION || data[0]== CW_MAGNETIC) {
					if (CWMCU_i2c_read(sensor, CW_ACCURACY, data, 1) >= 0) {
						data_event[5] = ((u32)id_check << 16) | (u32)data[0];
					}
					input_report_abs(sensor->input, CW_ABS_X, data_event[0]);
					input_report_abs(sensor->input, CW_ABS_Y, data_event[1]);
					input_report_abs(sensor->input, CW_ABS_Z, data_event[2]);
					input_report_abs(sensor->input, CW_ABS_TIMESTAMP, data_event[3]);
					input_report_abs(sensor->input, CW_ABS_MAG_ACCURACY, data_event[5]);
					input_sync(sensor->input);
				}
				else if (data[0] == CW_MAGNETIC_UNCALIBRATED) {
					sensor->uncalibratedMag[wakeup][0] = rawdata[0];
					sensor->uncalibratedMag[wakeup][1] = rawdata[1];
					sensor->uncalibratedMag[wakeup][2] = rawdata[2];
					//printk("--CWMCU-- handle[%d]: uncalib Mag => %u, %u, %u, time:%u\n",
					//	wakeup, rawdata[0], rawdata[1], rawdata[2], timestamp);
				}
				else if (data[0] == CW_MAGNETIC_UNCALIBRATED_BIAS) {
					id_check = CW_MAGNETIC_UNCALIBRATED + 32 * wakeup;
					data_event[0] = ((u32)id_check << 16) | sensor->uncalibratedMag[wakeup][0];
					data_event[1] = ((u32)id_check << 16) | sensor->uncalibratedMag[wakeup][1];
					data_event[2] = ((u32)id_check << 16) | sensor->uncalibratedMag[wakeup][2];
					data_event[5] = ((u32)id_check << 16) | (((u32)data[2] << 8) | (u32)data[1]);
					data_event[6] = ((u32)id_check << 16) | (((u32)data[4] << 8) | (u32)data[3]);
					data_event[7] = ((u32)id_check << 16) | (((u32)data[6] << 8) | (u32)data[5]);

					if (CWMCU_i2c_read(sensor, CW_ACCURACY, data, 1) >= 0) {
						data_event[8] = ((u32)id_check << 16) | (u32)data[0];
					}

					//printk("--CWMCU-- handle[%d]: uncalib Mag => %u, %u, %u, bias:%u, %u, %u, time:%u\n",
					//	wakeup, sensor->uncalibratedMag[wakeup][0], sensor->uncalibratedMag[wakeup][1], sensor->uncalibratedMag[wakeup][2],
					//	rawdata[0], rawdata[1], rawdata[2], timestamp);

					input_report_abs(sensor->input, CW_ABS_X, data_event[0]);
					input_report_abs(sensor->input, CW_ABS_Y, data_event[1]);
					input_report_abs(sensor->input, CW_ABS_Z, data_event[2]);
					input_report_abs(sensor->input, CW_ABS_X1, data_event[5]);
					input_report_abs(sensor->input, CW_ABS_Y1, data_event[6]);
					input_report_abs(sensor->input, CW_ABS_Z1, data_event[7]);
					input_report_abs(sensor->input, CW_ABS_MAG_ACCURACY, data_event[8]);
					input_report_abs(sensor->input, CW_ABS_TIMESTAMP, data_event[3]);
					input_sync(sensor->input);
				}
				else if (data[0] == CW_GYROSCOPE_UNCALIBRATED) {
					sensor->uncalibratedGyro[wakeup][0] = rawdata[0];
					sensor->uncalibratedGyro[wakeup][1] = rawdata[1];
					sensor->uncalibratedGyro[wakeup][2] = rawdata[2];
					//printk("--CWMCU-- handle[%d]: uncalib gyro => %u, %u, %u, time:%u\n",
					//	wakeup, rawdata[0], rawdata[1], rawdata[2], timestamp);

				}
				else if (data[0] == CW_GYROSCOPE_UNCALIBRATED_BIAS) {
					id_check = CW_GYROSCOPE_UNCALIBRATED + 32 * wakeup;
					data_event[0] = ((u32)id_check << 16) | sensor->uncalibratedGyro[wakeup][0];
					data_event[1] = ((u32)id_check << 16) | sensor->uncalibratedGyro[wakeup][1];
					data_event[2] = ((u32)id_check << 16) | sensor->uncalibratedGyro[wakeup][2];
					data_event[5] = ((u32)id_check << 16) | (((u32)data[2] << 8) | (u32)data[1]);
					data_event[6] = ((u32)id_check << 16) | (((u32)data[4] << 8) | (u32)data[3]);
					data_event[7] = ((u32)id_check << 16) | (((u32)data[6] << 8) | (u32)data[5]);

					//printk("--CWMCU-- handle[%d]: uncalib gyro => %u, %u, %u, bias:%u, %u, %u, time:%u\n",
					//	wakeup, sensor->uncalibratedGyro[wakeup][0], sensor->uncalibratedGyro[wakeup][1], sensor->uncalibratedGyro[wakeup][2],
					//	rawdata[0], rawdata[1], rawdata[2], timestamp);

					input_report_abs(sensor->input, CW_ABS_X, data_event[0]);
					input_report_abs(sensor->input, CW_ABS_Y, data_event[1]);
					input_report_abs(sensor->input, CW_ABS_Z, data_event[2]);
					input_report_abs(sensor->input, CW_ABS_X1, data_event[5]);
					input_report_abs(sensor->input, CW_ABS_Y1, data_event[6]);
					input_report_abs(sensor->input, CW_ABS_Z1, data_event[7]);
					input_report_abs(sensor->input, CW_ABS_TIMESTAMP, data_event[3]);
					input_sync(sensor->input);
					

				} else {
					//VERBOSE("--CWMCU-- stream[%d] id:%d timestamp=%u | rawdata:%u %u %u \n",wakeup,sensor_id, timestamp, rawdata[0],rawdata[1],rawdata[2]);

					if((1<<sensor_id) & SP_ENABLE_LIST) {
						spdata = (int)rawdata[0];
						print_sp_event(sensor_id,spdata);
                        //printk("CWMCU cwmcu_stream_mode_read 1 request 2s wakelock\n");
                        //wake_lock_timeout(&wakeup_event_wake_lock, 2*HZ);
					}
					input_report_abs(sensor->input, CW_ABS_X, data_event[0]);
					input_report_abs(sensor->input, CW_ABS_Y, data_event[1]);
					input_report_abs(sensor->input, CW_ABS_Z, data_event[2]);
					input_report_abs(sensor->input, CW_ABS_TIMESTAMP, data_event[3]);
	 				input_sync(sensor->input);
				}
				VERBOSE("--CWMCU-- stream[%d] id:%d data:%u %u %u | timestamp=%u\n",wakeup,sensor_id, rawdata[0],rawdata[1],rawdata[2],timestamp);

				//VERBOSE("--CWMCU-- handle[%d]: stream raw data => %2x,%2x,%2x,%2x,%2x,%2x,%2x,%2x,%2x,%2x,%2x\n",
				//	wakeup, data[0],data[1],data[2],data[3],data[4],data[5],data[6],data[7],data[8],data[9],data[10]);

				/* reset input event */
				input_report_abs(sensor->input, CW_ABS_X, 0xFF0000);
				input_report_abs(sensor->input, CW_ABS_Y, 0xFF0000);
				input_report_abs(sensor->input, CW_ABS_Z, 0xFF0000);
				input_report_abs(sensor->input, CW_ABS_X1, 0xFF0000);
				input_report_abs(sensor->input, CW_ABS_Y1, 0xFF0000);
				input_report_abs(sensor->input, CW_ABS_Z1, 0xFF0000);
	 			input_report_abs(sensor->input, CW_ABS_MAG_ACCURACY, 0xFF0000);
				input_sync(sensor->input);
			}
		} else {
			printk("--CWMCU-- handle[%d]: read stream event failed[i2c]!!\n", wakeup);
		}
	}


	if (CWMCU_i2c_read(sensor, stream_mode_onchange_reg_count, data, 1) >= 0) {
		onchange_q_count = data[0];
	}
	else {
		onchange_q_count = 0;
		printk("--CWMCU-- handle[%d]: read stream onchange event count failed[i2c]!!\n", wakeup);
	}
	VERBOSE("--CWMCU-- handle[%d]: onchange queue count:%d\n", wakeup, onchange_q_count);

	for (i = 0; i < onchange_q_count; i++) {
		/* read 11byte */
		if (CWMCU_i2c_read(sensor, stream_mode_onchange_reg_event, data, 11) >= 0) {
			//printk("--CWMCU-- handle[%d]: read stream onchange data = %d\n", wakeup, onchange_q_count);
			if (data[0] != CWMCU_NODATA) {
				id_check = data[0] + 32 * wakeup;
				data_event[0] = ((u32)id_check << 16) | (((u32)data[2] << 8) | (u32)data[1]);
				data_event[1] = ((u32)id_check << 16) | (((u32)data[4] << 8) | (u32)data[3]);
				data_event[2] = ((u32)id_check << 16) | (((u32)data[6] << 8) | (u32)data[5]);
				data_event[3] = (uint32_t)(data[10]<<24) | (uint32_t)(data[9]<<16) | (uint32_t)(data[8]<<8) | (uint32_t)data[7];
				rawdata[0] = (((u32)data[2]  << 8) | (u32)data[1]);
                rawdata[1] = (((u32)data[4]  << 8) | (u32)data[3]);
                rawdata[2] = (((u32)data[6]  << 8) | (u32)data[5]);
				timestamp = data_event[3];

				input_report_abs(sensor->input, CW_ABS_X, data_event[0]);
				input_report_abs(sensor->input, CW_ABS_Y, data_event[1]);
				input_report_abs(sensor->input, CW_ABS_Z, data_event[2]);
				input_report_abs(sensor->input, CW_ABS_TIMESTAMP, data_event[3]);
	 			input_sync(sensor->input);
				
				//VERBOSE("--CWMCU-- handle[%d]: stream onchange raw data => %2x,%2x,%2x,%2x,%2x,%2x,%2x,%2x,%2x,%2x,%2x\n",
				//	wakeup, data[0],data[1],data[2],data[3],data[4],data[5],data[6],data[7],data[8],data[9],data[10]);

				if(data[0]==CW_STEP_COUNTER) {
					sensor->step_counter = (uint32_t)data[4]<< 24 | (uint32_t)data[3]<< 16 | (uint32_t)data[2]<< 8 | data[1];
					VERBOSE("--CWMCU-- step counter=%d\n data[1-4]:%2x %2x %2x %2x", sensor->step_counter,data[1],data[2],data[3],data[4]);
				}
				
				if((1<<data[0]) & SP_ENABLE_LIST) {
					spdata = (int)rawdata[0];
					print_sp_event(data[0],spdata);
                    printk("CWMCU cwmcu_stream_mode_read 2 request 4s wakelock\n");
                    wake_lock_timeout(&wakeup_event_wake_lock, 4*HZ);
				}
				VERBOSE("--CWMCU-- stream[%d] onchange id:%d data:x=%u y=%u z=%u | time=%u\n",wakeup,id_check, rawdata[0],rawdata[1],rawdata[2],timestamp);
				

				/* reset input event */
				input_report_abs(sensor->input, CW_ABS_X, 0xFF0000);
				input_report_abs(sensor->input, CW_ABS_Y, 0xFF0000);
				input_report_abs(sensor->input, CW_ABS_Z, 0xFF0000);
				input_sync(sensor->input);
			}

		} else {
			printk("--CWMCU-- handle[%d]: read stream onchange event failed[i2c]!!\n",wakeup);
		}
	}


    return 1;
}


static int cwmcu_batch_mode_read(struct CWMCU_data *sensor, int wakeup)
{
    int i = 0;
    uint8_t data[20] = {0,};
    uint32_t data_event[6] = {0,};
	uint32_t rawdata[6] = {0,};

    int id_check=0;
	int batch_q_count = 0, onchange_q_count = 0;
	uint8_t batch_mode_reg_count, batch_mode_reg_event;
	uint8_t batch_mode_onchange_reg_count, batch_mode_onchange_reg_event;
	uint8_t mHandle = 0;

	uint32_t timestamp=0;
	int spdata = 0;


    if (sensor->mcu_mode == CW_BOOT || sensor->mcu_power == CW_BOOT) {
        return 0;
    }


	if(wakeup==0) {
		batch_mode_reg_count = CW_BATCH_NONWAKEUP_COUNT;
		batch_mode_reg_event = CW_BATCH_NONWAKEUP_EVENT;
		batch_mode_onchange_reg_count = CW_BATCH_NONWAKEUP_ONCHANGEQ_COUNT;
		batch_mode_onchange_reg_event = CW_BATCH_NONWAKEUP_ONCHANGEQ_EVENT;
		mHandle = CW_NONWAKEUP;
	}
	else if(wakeup==1) {
		batch_mode_reg_count = CW_BATCH_WAKEUP_COUNT;
		batch_mode_reg_event = CW_BATCH_WAKEUP_EVENT;
		batch_mode_onchange_reg_count = CW_BATCH_WAKEUP_ONCHANGEQ_COUNT;
		batch_mode_onchange_reg_event = CW_BATCH_WAKEUP_ONCHANGEQ_EVENT;
		mHandle = CW_WAKEUP;
	}
	else {
		printk("--CWMCU-- handle[%d]: cwmcu_batch_read() parameter wakeup is worng value.\n", wakeup);
		return 0;
	}

	/* read the count of batch queue */
	if (CWMCU_i2c_read(sensor, batch_mode_reg_count, data, 2) >= 0) {
		batch_q_count = ((uint16_t)data[1] << 8) | (uint16_t)data[0];
	}
	else {
		batch_q_count = 0;
		printk("--CWMCU-- handle[%d]: read batch event count failed[i2c]!!\n", wakeup);
	}

	if (CWMCU_i2c_read(sensor, batch_mode_onchange_reg_count, data, 1) >= 0) {
		onchange_q_count = data[0];
	}
	else {
		onchange_q_count = 0;
		printk("--CWMCU-- handle[%d]: read batch onchange event count failed[i2c]!!\n", wakeup);
	}
	VERBOSE("--CWMCU-- handle[%d]: batch queue count:%d onchange queue count:%d\n", wakeup, batch_q_count, onchange_q_count);





    //4,294,967,295
    for (i = 0; i < batch_q_count; i++) {
        if (CWMCU_i2c_read(sensor, batch_mode_reg_event, data, 11) >= 0) {
            /* check if there are no data from queue */
            if (data[0] != CWMCU_NODATA) {
                if (data[0] == CW_META_DATA) {
					id_check = data[0] + 32 * wakeup;
                    data_event[0] = ((u32)id_check << 16) | ((u32)data[5] << 8) | (u32)data[6];
                    sensor->flush_list[mHandle] &= ~(1<<data[6]);
                    VERBOSE("--CWMCU-- handle[%d]: META_DATA => %02x,%02x,%02x,%02x,%02x,%02x,%02x,%02x,%02x,%02x,%02x\n",
                        wakeup, data[0],data[1],data[2],data[3],data[4],data[5],data[6],data[7],data[8],data[9],data[10]);

					cwmcu_send_flush(id_check);
                    //input_report_abs(sensor->input, CW_ABS_Z, data_event[0]);
                    //input_sync(sensor->input);
                    //input_report_abs(sensor->input, CW_ABS_Z, 0xFF0000);
                    //input_sync(sensor->input);
				} else if(data[0] != CWMCU_NODATA) {
					id_check = data[0] + 32 * wakeup;
                    data_event[0] = ((u32)id_check << 16) | ((u32)data[2] << 8) | (u32)data[1];
                    data_event[1] = ((u32)id_check << 16) | ((u32)data[4] << 8) | (u32)data[3];
                    data_event[2] = ((u32)id_check << 16) | ((u32)data[6] << 8) | (u32)data[5];
					data_event[3] = (uint32_t)(data[10]<<24) | (uint32_t)(data[9]<<16) | (uint32_t)(data[8]<<8) | (uint32_t)data[7];
					rawdata[0] = (((u32)data[2]  << 8) | (u32)data[1]);
                	rawdata[1] = (((u32)data[4]  << 8) | (u32)data[3]);
                	rawdata[2] = (((u32)data[6]  << 8) | (u32)data[5]);
					timestamp = data_event[3];
                    //printk("--CWMCU-- batch[%d] timestamp=%u\n",wakeup, timestamp);
					if(timestamp < sensor->preCTimestame) {
						//printk("--CWMCU-- batch[%d] preCTimestame:%u timestamp=%u \n",wakeup, timestamp, sensor->preCTimestame);
						//timestamp = sensor->preCTSTimestame[CW_ACCELERATION] + sensor->report_period[wakeup][data[0]];
						sensor->preCTimestame = sensor->preCTSTimestame[CW_ACCELERATION] + 1;
					}
					else {
						sensor->preCTimestame = timestamp;
					}
					data_event[3] = timestamp;
					

					if(id_check == CW_ACCELERATION) {
						if(timestamp == 0 || timestamp == sensor->preCTSTimestame[CW_ACCELERATION] || timestamp < sensor->preCTSTimestame[CW_ACCELERATION]) {
							//printk("--CWMCU-- castorAcc_batch[%d] timestamp == preCTSTimestame | %u\n",wakeup,sensor->preCTSTimestame[CW_ACCELERATION]);
							timestamp = sensor->preCTSTimestame[CW_ACCELERATION] + sensor->report_period[wakeup][data[0]];
							sensor->preCTSTimestame[CW_ACCELERATION] = timestamp;
							data_event[3] = timestamp;
						}
						else
							sensor->preCTSTimestame[CW_ACCELERATION] = timestamp;
					}
					if(id_check == CW_MAGNETIC) {
						if(timestamp == 0 || timestamp == sensor->preCTSTimestame[CW_MAGNETIC] || timestamp < sensor->preCTSTimestame[CW_MAGNETIC]) {
							//printk("--CWMCU-- castorMag_batch[%d] timestamp == preCTSTimestame | %u\n",wakeup,sensor->preCTSTimestame[CW_MAGNETIC]);
							timestamp = sensor->preCTSTimestame[CW_MAGNETIC] + sensor->report_period[wakeup][data[0]];
							sensor->preCTSTimestame[CW_MAGNETIC] = timestamp;
							data_event[3] = timestamp;
						}
						else
							sensor->preCTSTimestame[CW_MAGNETIC] = timestamp;
					}
					if(id_check == CW_GYRO) {
						if(timestamp == 0 || timestamp == sensor->preCTSTimestame[CW_GYRO] || timestamp < sensor->preCTSTimestame[CW_GYRO]) {
							//printk("--CWMCU-- castorGyro_batch[%d] timestamp == preCTSTimestame | %u\n",wakeup,sensor->preCTSTimestame[CW_GYRO]);
							timestamp = sensor->preCTSTimestame[CW_MAGNETIC] + sensor->report_period[wakeup][data[0]];
							sensor->preCTSTimestame[CW_MAGNETIC] = timestamp;
							data_event[3] = timestamp;
						}
						else
							sensor->preCTSTimestame[CW_GYRO] = timestamp;
					}
					
					if((1<<data[0]) & SP_ENABLE_LIST) {
						spdata = (int)rawdata[0];
						print_sp_event(data[0],spdata);
                        //printk("CWMCU cwmcu_batch_mode_read 1 request 2s wakelock\n");
                        //wake_lock_timeout(&wakeup_event_wake_lock, 2*HZ);
					}
						

                    /* check flush event */
                    input_report_abs(sensor->input, CW_ABS_X, data_event[0]);
                    input_report_abs(sensor->input, CW_ABS_Y, data_event[1]);
                    input_report_abs(sensor->input, CW_ABS_Z, data_event[2]);
                    input_report_abs(sensor->input, CW_ABS_TIMESTAMP, data_event[3]);
                    input_sync(sensor->input);

                    //VERBOSE("--CWMCU-- handle[%d]: Batch raw data => %2x,%2x,%2x,%2x,%2x,%2x,%2x,%2x,%2x,%2x,%2x\n",
                    //    wakeup, data[0],data[1],data[2],data[3],data[4],data[5],data[6],data[7],data[8],data[9],data[10]);

					VERBOSE("--CWMCU-- batch[%d] id:%d data:%u %u %u | timestamp=%u\n",wakeup,id_check, rawdata[0],rawdata[1],rawdata[2],timestamp);

					input_report_abs(sensor->input, CW_ABS_X, 0xFF0000);
                    input_report_abs(sensor->input, CW_ABS_Y, 0xFF0000);
                    input_report_abs(sensor->input, CW_ABS_Z, 0xFF0000);
                    input_sync(sensor->input);
                }
            }
        } else {
            printk("--CWMCU-- handle[%d]: read batch event failed~!!\n", wakeup);
        }
    }



	for (i = 0; i < onchange_q_count; i++) {
		/* read 11byte */
		if (CWMCU_i2c_read(sensor, batch_mode_onchange_reg_event, data, 11) >= 0) {
			printk("--CWMCU-- handle[%d]: read batch onchange data = %d\n", wakeup, onchange_q_count);
			if (data[0] != CWMCU_NODATA) {
				id_check = data[0] + 32 * wakeup;
				data_event[0] = ((u32)id_check << 16) | (((u32)data[2] << 8) | (u32)data[1]);
				data_event[1] = ((u32)id_check << 16) | (((u32)data[4] << 8) | (u32)data[3]);
				data_event[2] = ((u32)id_check << 16) | (((u32)data[6] << 8) | (u32)data[5]);
				data_event[3] = (uint32_t)(data[10]<<24) | (uint32_t)(data[9]<<16) | (uint32_t)(data[8]<<8) | (uint32_t)data[7];
				rawdata[0] = (((u32)data[2]  << 8) | (u32)data[1]);
                rawdata[1] = (((u32)data[4]  << 8) | (u32)data[3]);
                rawdata[2] = (((u32)data[6]  << 8) | (u32)data[5]);
				timestamp = data_event[3];

				input_report_abs(sensor->input, CW_ABS_X, data_event[0]);
				input_report_abs(sensor->input, CW_ABS_Y, data_event[1]);
				input_report_abs(sensor->input, CW_ABS_Z, data_event[2]);
				input_report_abs(sensor->input, CW_ABS_TIMESTAMP, data_event[3]);
	 			input_sync(sensor->input);

				//VERBOSE("--CWMCU-- handle[%d]: batch onchange raw data => %2x,%2x,%2x,%2x,%2x,%2x,%2x,%2x,%2x,%2x,%2x\n",
				//	wakeup, data[0],data[1],data[2],data[3],data[4],data[5],data[6],data[7],data[8],data[9],data[10]);

				if(data[0]==CW_STEP_COUNTER) {
					sensor->step_counter = (uint32_t)data[4]<< 24 | (uint32_t)data[3]<< 16 | (uint32_t)data[2]<< 8 | data[1];
					VERBOSE("--CWMCU-- step counter=%d\n data[1-4]:%2x %2x %2x %2x", sensor->step_counter,data[1],data[2],data[3],data[4]);
				}
				
				if((1<<data[0]) & SP_ENABLE_LIST) {
					spdata = (int)rawdata[0];
					print_sp_event(data[0],spdata);
                    //printk("CWMCU cwmcu_batch_mode_read 2 request 2s wakelock\n");
                    //wake_lock_timeout(&wakeup_event_wake_lock, 2*HZ);
				}
				VERBOSE("--CWMCU-- batch[%d] onchange id:%d data:x=%u y=%u z=%u | time=%u\n",wakeup,id_check, rawdata[0],rawdata[1],rawdata[2],timestamp);

				/* reset input event */
				input_report_abs(sensor->input, CW_ABS_X, 0xFF0000);
				input_report_abs(sensor->input, CW_ABS_Y, 0xFF0000);
				input_report_abs(sensor->input, CW_ABS_Z, 0xFF0000);
				input_sync(sensor->input);
			}

		} else {
			printk("--CWMCU-- handle[%d]: read batch onchange event failed[i2c]!!\n",wakeup);
		}
	}



    return 1;
}




static int cwmcu_error_log_read(struct CWMCU_data *sensor)
{
    uint8_t data[32] = {0};
    char log_message[32];
    int data_count = 0;
    int i = 0;

    if (sensor->mcu_mode == CW_BOOT || sensor->mcu_power == CW_BOOT) {
        return 0;
    }

    if (CWMCU_i2c_read(sensor, CW_ERROR_COUNT_GET, data, 1) >= 0) {
        data_count = data[0];
        printk("--CWMCU-- read error_log: count = %d\n", data_count);
        for (i = 0; i < data_count; i++) {
            /* read 32bytes */
            if (CWMCU_i2c_read(sensor, CW_ERROR_INFO_GET, data, 32) >= 0) {
                    memcpy(log_message,data,32);
                    printk("--CWMCU-- error_log => %s\n", log_message);
            } else {
                printk("--CWMCU-- read error_log failed.\n");
            }
        }
    }

    return 1;
}

static int cwmcu_debug_log_read(struct CWMCU_data *sensor)
{
    uint8_t data[32] = {0};
    char log_message[32];
    int data_count = 0;
    int i = 0;

    if (sensor->mcu_mode == CW_BOOT || sensor->mcu_power == CW_BOOT) {
        return 0;
    }

    if (CWMCU_i2c_read(sensor, CW_DEBUG_COUNT_GET, data, 1) >= 0) {
        data_count = data[0];
        printk("--CWMCU-- read debug_log: count = %d\n", data_count);
        for (i = 0; i < data_count; i++) {
            /* read 32bytes */
            memset(data,0,32);
            memset(log_message,0,32);
            if (CWMCU_i2c_read(sensor, CW_DEBUG_INFO_GET, data, 32) >= 0) {
                memcpy(log_message,data,32);
                printk("--CWMCU-- debug_log => %s\n", log_message);
            } else {
                printk("--CWMCU-- read debug_log failed.\n");
            }
        }
    }

    return 1;
}



#ifndef CWMCU_INTERRUPT
static void cwmcu_check_sensor_update(void)
{
    int64_t tvusec, temp = 0;
    unsigned int tvsec;

    do_gettimeofday(&sensor->now);
    tvsec = sensor->now.tv_sec;
    tvusec = sensor->now.tv_usec;

    temp = (int64_t)(tvsec * 1000000LL) + tvusec;

    sensor->update_list[CW_NONWAKEUP] = sensor->enabled_list[CW_NONWAKEUP];
	sensor->update_list[CW_WAKEUP] = sensor->enabled_list[CW_WAKEUP];
}
#endif


static int cwmcu_check_report_list(void)
{
	uint8_t data[4] = {0};
	uint8_t retry = 0;

	for (retry = 0; retry < 3;retry++) {
		if (CWMCU_i2c_read(sensor, CW_REPORT_SENSORS_LIST, data, 4) >= 0) {
			sensor->sensors_report_list = (u32)data[3] << 24 | (u32)data[2] << 16 | (u32)data[1] << 8 | (u32)data[0];
			printk(KERN_DEBUG "--CWMCU-- sensor->sensors_report_list = 0x%x\n", sensor->sensors_report_list);
			return 0;
		}
		usleep_range(20000, 20000);
	}
	return retry;
}

static int CWMCU_read(struct CWMCU_data *sensor)
{
    int id_check = 0, report_id = 0;
    uint8_t data[20] = {0};
    uint32_t data_event[10] = {0};
	uint8_t MCU_REG = 0;
	
	uint32_t rawdata[6] = {0,};
	uint32_t timestamp = 0;
	int spdata = 0;
    

    if ((sensor->mcu_mode == CW_BOOT) || sensor->mcu_status == CW_INVAILD || sensor->mcu_power == CW_BOOT) {
        /* it will not get data if status is bootloader mode */
        return 0;
    }

#ifndef CWMCU_INTERRUPT
    cwmcu_check_sensor_update();
#endif


	cwmcu_check_report_list();
    if ( sensor->sensors_report_list & CWMCU_MEM_SENSORS_LIST ) {
        for (id_check = 0; id_check < CW_SENSORS_ID_END; id_check++) {
			if( sensor->enabled_list[CW_NONWAKEUP] & (1<<id_check) || sensor->enabled_list[CW_WAKEUP] & (1<<id_check) ) {
                switch (id_check) {
                case CW_ACCELERATION:
                case CW_MAGNETIC:
                case CW_GYRO:
                case CW_LIGHT:
                //case CW_PROXIMITY:
                case CW_PRESSURE:
                case CW_ORIENTATION:
                case CW_ROTATIONVECTOR:
                case CW_LINEARACCELERATION:
                case CW_GRAVITY:
                case CW_GAME_ROTATION_VECTOR:
                case CW_GEOMAGNETIC_ROTATION_VECTOR:
                //case CW_PROXIMITY_GESTURE:
                //case CW_LIGHT_RGB:
				
					MCU_REG = CWMCU_I2C_SENSORS_REG_START+id_check;
					
					report_id = id_check;
					if((1<<id_check) & sensor->enabled_list[CW_WAKEUP]) {
						report_id = id_check + 32;
					}
					
					if (CWMCU_i2c_read(sensor, CW_MCU_TIMESTAMP, data, 4) >= 0) {
						timestamp = (uint32_t)(data[3]<<24) | (uint32_t)(data[2]<<16) | (uint32_t)(data[1]<<8) | (uint32_t)data[0];
						printk("--CWMCU-- mcu timestamp:%u\n", timestamp);
					}
					
	
                    /* read 6byte */
                    if (CWMCU_i2c_read(sensor, MCU_REG, data, 6) >= 0) {
                        data_event[0] = ((u32)report_id << 16) | (((u32)data[1] << 8) | (u32)data[0]);
                        data_event[1] = ((u32)report_id << 16) | (((u32)data[3] << 8) | (u32)data[2]);
                        data_event[2] = ((u32)report_id << 16) | (((u32)data[5] << 8) | (u32)data[4]);

                        VERBOSE("--CWMCU--Normal id=%d data -> x = %d, y = %d, z = %d\n"
                                    , report_id
                                    , (int16_t)((u32)data[1] << 8) | (u32)data[0]
                                    , (int16_t)((u32)data[3] << 8) | (u32)data[2]
                                    , (int16_t)((u32)data[5] << 8) | (u32)data[4]
                                    );
									
						if (CWMCU_i2c_read(sensor, CW_MCU_TIMESTAMP, data, 4) >= 0) {
							data_event[3] = (uint32_t)(data[3]<<24) | (uint32_t)(data[2]<<16) | (uint32_t)(data[1]<<8) | (uint32_t)data[0];
							printk("--CWMCU-- mcu timestamp:%u\n", timestamp);
						}
						rawdata[0] = (((u32)data[1]  << 8) | (u32)data[0]);
                		rawdata[1] = (((u32)data[3]  << 8) | (u32)data[2]);
                		rawdata[2] = (((u32)data[5]  << 8) | (u32)data[4]);
						timestamp = data_event[3];
						
						if(id_check == CW_ORIENTATION || id_check== CW_MAGNETIC) {
							if (CWMCU_i2c_read(sensor, CW_ACCURACY, data, 1) >= 0) {
								data_event[5] = ((u32)report_id << 16) | (u32)data[0];
							}
							input_report_abs(sensor->input, CW_ABS_X, data_event[0]);
							input_report_abs(sensor->input, CW_ABS_Y, data_event[1]);
							input_report_abs(sensor->input, CW_ABS_Z, data_event[2]);
							input_report_abs(sensor->input, CW_ABS_TIMESTAMP, data_event[3]);
							input_report_abs(sensor->input, CW_ABS_MAG_ACCURACY, data_event[5]);
							input_sync(sensor->input);
						} else {
							//VERBOSE("--CWMCU-- stream[%d] id:%d timestamp=%u | rawdata:%u %u %u \n",wakeup,id_check, timestamp, rawdata[0],rawdata[1],rawdata[2]);

							if((1<<id_check) & SP_ENABLE_LIST) {
								spdata = (int)rawdata[0];
								print_sp_event(id_check,spdata);
                                //printk("CWMCU CWMCU_read 1 request 2s wakelock\n");
                                //wake_lock_timeout(&wakeup_event_wake_lock, 2*HZ);
							}
							input_report_abs(sensor->input, CW_ABS_X, data_event[0]);
							input_report_abs(sensor->input, CW_ABS_Y, data_event[1]);
							input_report_abs(sensor->input, CW_ABS_Z, data_event[2]);
							input_report_abs(sensor->input, CW_ABS_TIMESTAMP, data_event[3]);
							input_sync(sensor->input);
						}
				


                        /* reset x,y,z */
                        input_report_abs(sensor->input, CW_ABS_X, 0xFF0000);
                        input_report_abs(sensor->input, CW_ABS_Y, 0xFF0000);
                        input_report_abs(sensor->input, CW_ABS_Z, 0xFF0000);
                        input_report_abs(sensor->input, CW_ABS_MAG_ACCURACY, 0xFF0000);
                        input_sync(sensor->input);

                    } else {
                        printk("--CWMCU-- CWMCU_i2c_read error 0x%x~!!!\n", CWMCU_I2C_SENSORS_REG_START+id_check);
                    }
                    break;

                case CW_MAGNETIC_UNCALIBRATED:
                case CW_GYROSCOPE_UNCALIBRATED:
                    /* read 12byte */
                    if (CWMCU_i2c_read(sensor, CWMCU_I2C_SENSORS_REG_START+id_check, data, 12) >= 0) {
                        data_event[0] = ((u32)report_id << 16) | (((u32)data[1] << 8) | (u32)data[0]);
                        data_event[1] = ((u32)report_id << 16) | (((u32)data[3] << 8) | (u32)data[2]);
                        data_event[2] = ((u32)report_id << 16) | (((u32)data[5] << 8) | (u32)data[4]);
                        data_event[3] = ((u32)report_id << 16) | (((u32)data[7] << 8) | (u32)data[6]);
                        data_event[4] = ((u32)report_id << 16) | (((u32)data[9] << 8) | (u32)data[8]);
                        data_event[5] = ((u32)report_id << 16) | (((u32)data[11] << 8) | (u32)data[10]);

                        VERBOSE("--CWMCU--Normal id=%d data -> x = %d, y = %d, z = %d, x_bios = %d, y_bios = %d, z_bios = %d,\n"
                                    , report_id
                                    , (int16_t)((u32)data[1] << 8) | (u32)data[0]
                                    , (int16_t)((u32)data[3] << 8) | (u32)data[2]
                                    , (int16_t)((u32)data[5] << 8) | (u32)data[4]
                                    , (int16_t)((u32)data[7] << 8) | (u32)data[6]
                                    , (int16_t)((u32)data[9] << 8) | (u32)data[8]
                                    , (int16_t)((u32)data[11] << 8) | (u32)data[10]
                                );
						
                        if (CWMCU_i2c_read(sensor, CW_ACCURACY, data, 1) >= 0) {
                            data_event[6] = ((u32)id_check << 16) | (u32)data[0];
                        }
						
						if (CWMCU_i2c_read(sensor, CW_MCU_TIMESTAMP, data, 4) >= 0) {
							data_event[7] = (uint32_t)(data[3]<<24) | (uint32_t)(data[2]<<16) | (uint32_t)(data[1]<<8) | (uint32_t)data[0];
							VERBOSE("--CWMCU-- mcu timestamp:%u\n", timestamp);
						}
						
                        input_report_abs(sensor->input, CW_ABS_X, data_event[0]);
                        input_report_abs(sensor->input, CW_ABS_Y, data_event[1]);
                        input_report_abs(sensor->input, CW_ABS_Z, data_event[2]);
                        input_report_abs(sensor->input, CW_ABS_X1, data_event[3]);
                        input_report_abs(sensor->input, CW_ABS_Y1, data_event[4]);
                        input_report_abs(sensor->input, CW_ABS_Z1, data_event[5]);
                        input_report_abs(sensor->input, CW_ABS_MAG_ACCURACY, data_event[6]);
						input_report_abs(sensor->input, CW_ABS_TIMESTAMP, data_event[7]);
                        input_sync(sensor->input);

                        input_report_abs(sensor->input, CW_ABS_X, 0xFF0000);
                        input_report_abs(sensor->input, CW_ABS_Y, 0xFF0000);
                        input_report_abs(sensor->input, CW_ABS_Z, 0xFF0000);
                        input_report_abs(sensor->input, CW_ABS_X1, 0xFF0000);
                        input_report_abs(sensor->input, CW_ABS_Y1, 0xFF0000);
                        input_report_abs(sensor->input, CW_ABS_Z1, 0xFF0000);
                        input_report_abs(sensor->input, CW_ABS_MAG_ACCURACY, 0xFF0000);
                        input_sync(sensor->input);
                    }
                    break;
                case CW_STEP_COUNTER:
                    /* read 6byte */
                    if (CWMCU_i2c_read(sensor, CWMCU_I2C_SENSORS_REG_START+id_check, data, 4) >= 0) {
                        data_event[0] = ((u32)report_id << 16) | (((u32)data[1] << 8) | (u32)data[0]);
                        data_event[1] = ((u32)report_id << 16) | (((u32)data[3] << 8) | (u32)data[2]);

                        sensor->step_counter = (uint32_t)data[3]<< 24 | (uint32_t)data[2]<< 16 | (uint32_t)data[1]<< 8 | data[0];
						
						if (CWMCU_i2c_read(sensor, CW_MCU_TIMESTAMP, data, 4) >= 0) {
							data_event[3] = (uint32_t)(data[3]<<24) | (uint32_t)(data[2]<<16) | (uint32_t)(data[1]<<8) | (uint32_t)data[0];
							VERBOSE("--CWMCU-- mcu timestamp:%u\n", timestamp);
						}

                        input_report_abs(sensor->input, CW_ABS_X, data_event[0]);
                        input_report_abs(sensor->input, CW_ABS_Y, data_event[1]);
						input_report_abs(sensor->input, CW_ABS_TIMESTAMP, data_event[3]);
                        input_sync(sensor->input);

                        VERBOSE("--CWMCU--Normal id=%d data -> x = %d, y = %d, z = 0\n"
                                    , report_id
                                    , (int16_t)((u32)data[1] << 8) | (u32)data[0]
                                    , (int16_t)((u32)data[3] << 8) | (u32)data[2]);
                    } else {
                        printk("--CWMCU-- CWMCU_i2c_read error 0x%x~!!!\n", CWMCU_I2C_SENSORS_REG_START+id_check);
                    }
                    break;
                default:
                    break;

                }
            }
        }
    }

    return 0;
}



/*==========sysfs node=====================*/

static ssize_t active_set(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    int enabled = 0;
    int sensors_id = 0;
    u8 data[2] = {0,};
	
	uint8_t mSensorId, mHandle;

    sscanf(buf, "%d %d\n", &sensors_id, &enabled);


	mHandle   = (uint8_t)(sensors_id/32);
	mSensorId = (uint8_t)(sensors_id%32);
	sensor->enabled_list[mHandle] &= ~(1<<mSensorId);
    sensor->enabled_list[mHandle] |= ((uint32_t)enabled)<<mSensorId;

    /* clean timeout value if sensor turn off */
    if (enabled == 0) {
        sensor->sensor_timeout[mHandle][sensors_id] = 0;
    }

    if ((sensor->mcu_mode == CW_BOOT) || sensor->mcu_status == CW_INVAILD || sensor->mcu_power == CW_BOOT) {
        return count;
    }

	data[0] = (uint8_t)sensors_id;
	data[1] = (uint8_t)enabled;
	//printk( "--CWMCU-- active sensor data: %d %d\n",data[0], data[1]);
	if(CWMCU_write_i2c_block(sensor, CW_HOST_ACTIVATE, data, 2) < 0) {
		printk( "--CWMCU-- active sensor error. [i2c]\n");
	}

    printk( "--CWMCU-- active sensor sensors_id=%d enable=%d  enable_list[%d]=0x%08x\n", sensors_id, enabled, mHandle, sensor->enabled_list[mHandle]);

    return count;
}

static ssize_t active_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	int len = 0;
	len += sprintf(buf + len,
                        "enabled_list_nonwp       : 0x%x\n" \
                        "enabled_list_wp          : 0x%x\n", \
                        sensor->enabled_list[CW_NONWAKEUP],
						sensor->enabled_list[CW_WAKEUP]);
						
    return len;
}

static ssize_t interval_show(struct device *dev, struct device_attribute *attr, char *buf)
{
#ifdef CWMCU_INTERRUPT
    int id,hd;
    uint8_t mHandle, mSensorId;
    uint8_t rData[2],wData[2];
    uint8_t sensor_report_rate[CW_HANDLE_ID_END][CW_SENSORS_ID_END];
    char line[512] = {0}; 

    memset(sensor_report_rate,0,sizeof(uint8_t)*CW_SENSORS_ID_END*CW_HANDLE_ID_END);
    for(hd=0;hd<CW_HANDLE_ID_END;hd++) {
        mHandle = (uint8_t)hd;
        wData[1] = mHandle;
        for(id=0;id<CW_SENSORS_ID_END;id++) {
            mSensorId = (uint8_t)id;
            wData[0] = mSensorId;
            if(CWMCU_write_i2c_block(sensor, CW_SENSOR_DELAY_ID_SET, wData, 2) >= 0) {
                if (CWMCU_i2c_read(sensor, CW_SENSOR_DELAY_GET, rData, 1) >= 0) {
                    sensor_report_rate[hd][id] = rData[0];
                } else {
                    printk("--CWMCU-- interval_show()=> Get sensor report rate error. sensorId=%d handle=%d\n",mSensorId,mHandle);
                }
            } else {
                printk("--CWMCU-- interval_show()=> Set sensor ID error. sensorId=%d handle=%d\n",mSensorId,mHandle);
            }
        }
    }

    for(hd=0;hd<CW_HANDLE_ID_END;hd++) {
        for(id=0;id<CW_SENSORS_ID_END;id++) {
            printk("--CWMCU-- sensor report rate(ms)  sensor[%d][%d] => %d\n",hd,id, sensor_report_rate[hd][id]);
            sprintf(line+hd*9*(CW_SENSORS_ID_END)+id*9, "%d-%02d:%3d\n", hd, id, sensor_report_rate[hd][id]);
        }
    }

    return snprintf(buf, 512, "%s", line);
#else
    return snprintf(buf, 16, "%d\n", CWMCU_POLL_INTERVAL);
#endif
}

static ssize_t interval_set(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    int val = 0;
    int sensors_id = 0;
	uint32_t timeout = 0;
    uint8_t delay_ms = 0;
    uint8_t data[7];
	
	uint8_t mHandle, mSensorId;


    sscanf(buf, "%d %d\n", &sensors_id , &val);
    if (val < CWMCU_POLL_MIN)
        val = CWMCU_POLL_MIN;


	mHandle   = (uint8_t)(sensors_id/32);
	mSensorId = (uint8_t)(sensors_id%32);
    sensor->report_period[mHandle][mSensorId] = val;

    if ((sensor->mcu_mode == CW_BOOT) || sensor->mcu_status == CW_INVAILD || sensor->mcu_power == CW_BOOT) {
        return count;
    }

    delay_ms = (uint8_t)val;

	//id, flag, minDelay, timeout
    data[0] = (uint8_t)sensors_id;
    data[1] = mHandle;
    data[2] = delay_ms;
	memcpy(data+3,&timeout,sizeof(uint32_t));
	if(CWMCU_write_i2c_multiple(sensor, CW_HOST_BATCH, data, 7) < 0) {
		printk("--CWMCU-- interval set fail. [i2c]\n");
	}

    printk( "--CWMCU-- sensors_id=%d delay_ms=%d\n", sensors_id, delay_ms);
    return count;
}

static ssize_t batch_set(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    int sensors_id = 0;
    int delay_ms = 0;
    int timeout = 0;
    int flag = -1;
    uint8_t data[7] = {0,};
    int id;

	uint8_t mHandle, mSensorId;
	

    printk("--CWMCU-- %s in~!!\n", __func__);

    sscanf(buf, "%d %d %d %d\n", &sensors_id, &flag, &delay_ms, &timeout);
    printk("--CWMCU-- sensors_id = %d, handle = %d, delay_ms = %d, timeout = %d\n", sensors_id, flag, delay_ms, timeout);


	mHandle   = (uint8_t)(sensors_id/32);
	mSensorId = (uint8_t)(sensors_id%32);
		
    sensor->sensor_timeout[mHandle][mSensorId] = (uint32_t)timeout;
    sensor->report_period[mHandle][mSensorId] = (uint8_t)delay_ms;

    if ((sensor->mcu_mode == CW_BOOT) || sensor->mcu_status == CW_INVAILD || sensor->mcu_power == CW_BOOT) {
        return count;
    }

    if (timeout > 0) {
        sensor->batched_list[mHandle] |= ((uint32_t)1<<mSensorId);
        sensor->batch_timeout[mHandle] = timeout;
    } else {
        sensor->batched_list[mHandle] &= ~((uint32_t)1<<mSensorId);
    }

    for(id=0;id<CW_SENSORS_ID_END;id++) {
        if(sensor->batched_list[mHandle] & (1<<id)) {
            if(sensor->sensor_timeout[mHandle][id] < sensor->batch_timeout[mHandle])
                sensor->batch_timeout[mHandle] = sensor->sensor_timeout[mHandle][id];
        }
    }

	//id, flag, minDelay, timeout
    data[0] = (uint8_t)sensors_id;
    data[1] = mHandle;
    data[2] = (uint8_t)delay_ms;
	memcpy(data+3,&timeout,sizeof(uint32_t));
	if(CWMCU_write_i2c_block(sensor, CW_HOST_BATCH, data, 7) < 0) {
		printk("--CWMCU-- batch set fail. [i2c]\n");
	}

    printk( "--CWMCU-- BatchSet=> id=%d, timeout=%d, delay_ms=%d, batch_timeout[%d]=%d\n",
            sensors_id, timeout, delay_ms, mHandle, sensor->batch_timeout[mHandle]);

    return count;
}

static ssize_t batch_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	int len = 0;
	len += sprintf(buf + len,
                        "batched_list_nonwp       : 0x%08x\n" \
                        "batched_list_wp          : 0x%08x\n" \
						"batch_timeout_nonwp      : %u\n" \
						"batch_timeout_wp         : %u\n", \
                        sensor->batched_list[CW_NONWAKEUP],
						sensor->batched_list[CW_WAKEUP],
						sensor->batch_timeout[CW_NONWAKEUP],
						sensor->batch_timeout[CW_WAKEUP]);
				
    return len;
}

static ssize_t flush_set(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    int sensors_id = 0;
	uint8_t data[2] = {0,};
	
	uint8_t mHandle, mSensorId;

    

    printk("--CWMCU-- %s in\n", __func__);

    sscanf(buf, "%d\n", &sensors_id);
    printk("--CWMCU-- flush => sensors_id = %d \n", sensors_id);
	
		
	mHandle   = (uint8_t)(sensors_id/32);
	mSensorId = (uint8_t)(sensors_id%32);
	sensor->flush_list[mHandle] &= ~(1<<mSensorId);
    sensor->flush_list[mHandle] |= (uint32_t)1<<mSensorId;


    if ((sensor->mcu_mode == CW_BOOT) || sensor->mcu_status == CW_INVAILD || sensor->mcu_power == CW_BOOT) {
        return count;
    }

	//id, flag, minDelay, timeout
    data[0] = mSensorId;
    data[1] = mHandle;
	if(CWMCU_write_i2c_block(sensor, CW_BATCH_FLUSH, data, 2) < 0) {
		printk("--CWMCU-- batch set fail. [i2c]\n");
	}

	//cwmcu_send_flush(sensors_id);


    return count;
}

static ssize_t flush_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	int len = 0;
	len += sprintf(buf + len,
                        "flush_list_nonwp       : 0x%08x\n"  \
                        "flush_list_wp          : 0x%08x\n", \
                        sensor->flush_list[CW_NONWAKEUP],
						sensor->flush_list[CW_WAKEUP]);
				
    return len;
}

#if 0
static int set_firmware_update_i2(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	int intsize = sizeof(int);
	memcpy(&sensor->cw_i2c_rw, buf, intsize);
	memcpy(&sensor->cw_i2c_len, &buf[4], intsize);
	memcpy(sensor->cw_i2c_data, &buf[8], sensor->cw_i2c_len);

	return count;
}

static int get_firmware_update_i2(struct device *dev, struct device_attribute *attr, char *buf)
{
	int status = 0;
	if (sensor->cw_i2c_rw) {
		if (CWMCU_write_serial(sensor->cw_i2c_data, sensor->cw_i2c_len) < 0) {
			status = -1;
		}
		memcpy(buf, &status, sizeof(int));
		return 4;
	} else {
		if (CWMCU_read_serial(sensor->cw_i2c_data, sensor->cw_i2c_len) < 0) {
			status = -1;
			memcpy(buf, &status, sizeof(int));
			return 4;
		}
		memcpy(buf, &status, sizeof(int));
		memcpy(&buf[4], sensor->cw_i2c_data, sensor->cw_i2c_len);
		return 4+sensor->cw_i2c_len;
	}
	return  0;
}
#endif

static ssize_t firmware_update_cmd_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("--CWMCU-- %s in\n", __func__);
	return snprintf(buf, 8, "%d\n", sensor->fwupdate_status);
}

static ssize_t firmware_update_cmd_set(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    bool ret = false;
    sscanf(buf, "%d\n", &sensor->cmd);
    printk("CWMCU cmd=%d \n", sensor->cmd);

    switch (sensor->cmd) {

    case CHANGE_TO_BOOTLOADER_MODE:
		printk("CWMCU CHANGE_TO_BOOTLOADER_MODE\n");
        sensor->fwupdate_status = 1;
        wake_lock_timeout(&fwupdate_wake_lock, 20*HZ);
        disable_irq(sensor->client->irq);

        sensor->mcu_mode = CW_BOOT;
        CWMCU_reset_mcu(false);
        sensor->client->addr = 0x18;

        if (!lpc5400x_probebus(sensor->client, SL_I2C1, sensor->irq_gpio)) {
            printk( "CWMCU Firmware update bus detection failed!\n");
            sensor->fwupdate_status = -1;
        }

        break;

    case MCU_IMAGE_WRITE:
        printk("CWMCU WRITE MCU IMAGE\n");

        if(sensor->fwupdate_status == -1) {
            printk("CWMCU check firmware update status error.\n");
            break;
        }
        ret = lpc5400x_updateFirmware(sensor->client, sensor->fw_name);

        sensor->client->addr = 0x3a;
/****zhujp2 add 2015.01.29*******/

        if (ret == true) {
            sensor->fwupdate_status = 2;
            sensor->mcu_mode = CW_NORMAL;
            sensor->mcu_status = CW_VAILD;
            usleep_range(190000, 200000);
            get_mag_vendor();
            enable_irq(sensor->client->irq);
        } else {
            printk("CWMCU write firmware image error.\n");
            sensor->fwupdate_status = -1;
            sensor->mcu_mode = CW_BOOT;
        }
/****zhujp2 add 2015.01.29 end*******/

/***********************zhujp2 delete 2015.01.29
        if (ret == true) {
            sensor->fwupdate_status = 2;
            sensor->mcu_mode = CW_NORMAL;
        } else {
            printk("CWMCU write firmware image error.\n");
            sensor->fwupdate_status = -1;
            sensor->mcu_mode = CW_BOOT;
        }
        if (cwmcu_get_version())
            sensor->mcu_status = CW_INVAILD;
        else {
            sensor->mcu_status = CW_VAILD;
            enable_irq(sensor->client->irq);
        }
***********************/
        break;
    default:
        break;
    }
    return count;
}

static ssize_t set_mcu_cmd(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    u8 data[40] = {0,};
    s16 data_buff[3] = {0,};
    u8 regValue;
    u8 McuReg = 0x3A;
    int i;
	int inValueA, inValueB;
	u8  cmData[6] = {0};

    sscanf(buf, "%d %x %d\n", &sensor->cmd, &sensor->addr, &sensor->value);
	sscanf(buf, "%d %d %d\n", &sensor->cmd, &inValueA, &inValueB);
    McuReg = (u8)sensor->addr;
    regValue = (u8)sensor->value;
	cmData[0] = inValueA;
    cmData[1] = inValueB;
    printk(KERN_DEBUG "CWMCU cmd=%d addr=0x%x value=%d\n", sensor->cmd, McuReg, sensor->value);

    switch (sensor->cmd) {

    case MCU_REGISTER_WRITE_CTRL:
        printk("CWMCU MCU_REGISTER_WRITE_CTRL\n");
        if (CWMCU_i2c_write(sensor, McuReg, &regValue, 1) < 0) {
            printk("--CWMCU-- Write MCU Reg error.\n");
        } else {
            printk("--CWMCU-- Write MCU Reg=0x%x: value=%d\n",McuReg,regValue);
        }
        break;

    case MCU_REGISTER_READ_CTRL:
        printk("CWMCU MCU_REGISTER_READ_CTRL\n");
        if (CWMCU_i2c_read(sensor, McuReg, data, regValue) < 0) {
            printk("--CWMCU-- Read MCU Reg error.\n");
        } else {
			for(i=0;i<regValue;i++) {
				printk("--CWMCU-- Read Data[%d]: %x\n",i,data[i]);
			}
        }
        break;

    case CHECK_ACC_DATA:
        printk(KERN_DEBUG "CWMCU CHECK_ACC_DATA\n");
        if (CWMCU_i2c_read(sensor, CW_READ_ACCELERATION, data, 6) >= 0) {
            data_buff[0] = (s16)(((u16)data[1] << 8) | (u16)data[0]);
            data_buff[1] = (s16)(((u16)data[3] << 8) | (u16)data[2]);
            data_buff[2] = (s16)(((u16)data[5] << 8) | (u16)data[4]);

            printk("--CWMCU-- ACC_DATA: x = %d, y = %d, z = %d\n",
                data_buff[0], data_buff[1], data_buff[2]);
        }
        break;

    case CHECK_MAG_DATA:
        printk(KERN_DEBUG "CWMCU CHECK_MAG_DATA\n");
        if (CWMCU_i2c_read(sensor, CW_READ_MAGNETIC, data, 6) >= 0) {
            data_buff[0] = (s16)(((u16)data[1] << 8) | (u16)data[0]);
            data_buff[1] = (s16)(((u16)data[3] << 8) | (u16)data[2]);
            data_buff[2] = (s16)(((u16)data[5] << 8) | (u16)data[4]);

            printk("--CWMCU-- MAG_DATA: x = %d, y = %d, z = %d\n",
                data_buff[0], data_buff[1], data_buff[2]);
        }
        break;

    case CHECK_GYRO_DATA:
        printk(KERN_DEBUG "CWMCU CHECK_GYRO_DATA\n");
        if (CWMCU_i2c_read(sensor, CW_READ_GYRO, data, 6) >= 0) {
            data_buff[0] = (s16)(((u16)data[1] << 8) | (u16)data[0]);
            data_buff[1] = (s16)(((u16)data[3] << 8) | (u16)data[2]);
            data_buff[2] = (s16)(((u16)data[5] << 8) | (u16)data[4]);

            printk("--CWMCU-- GYRO_DATA: d[0-6] = %2x %2x %2x %2x %2x %2x\n",
                data[0],data[1],data[2],data[3],data[4],data[5]);
        }
        break;

	case SET_SENSORHUB_DATA_MODE:
        printk("CWMCU SET_SENSORHUB_DATA_MODE\n");

		if(sensor->addr == 1)
			sensor->sensorhub_data_mode |= sensor->value;
		if(sensor->addr == 0)
			sensor->sensorhub_data_mode &= (~sensor->value);

		printk("CWMCU sensor->sensorhub_data_mode = %d\n",sensor->sensorhub_data_mode);
        break;

	case WDT_FEED_STATUS:
        printk("CWMCU CW_WDT_FEED_STATUS\n");
        if (CWMCU_i2c_write(sensor, CW_WDT_FEED_STATUS, &regValue, 1) < 0) {
            printk("--CWMCU-- Write MCU CW_WDT_FEED_STATUS error.\n");
        }
        break;

    case MCU_DBG_INT_SWITCH:
        printk("CWMCU CW_MCU_DBG_INT_SWITCH\n");
        if (CWMCU_i2c_write(sensor, CW_MCU_DBG_INT_SWITCH, &regValue, 1) < 0) {
            printk("CWMCU Write CW_MCU_DBG_INT_SWITCH error.\n");
        }
        break;

    case SET_PROXIMITY_THRESHOLD:
        printk("CWMCU CW_SET_PROXIMITY_THRESHOLD\n");
        printk("CWMCU set threshold: %d %d\n", cmData[0],cmData[1]);
        if (CWMCU_write_i2c_block(sensor, CW_SET_PROXIMITY_THRESHOLD, cmData, 2) < 0) {
            printk("--CWMCU-- Write MCU CW_SET_PROXIMITY_THRESHOLD error.\n");
        }
        break;
		
	case SET_INTERRUPT_ENABLE:
#ifdef CONFIG_SNS_HUB_DEBUG
        printk("CWMCU SET_INTERRUPT_ENABLE\n");
        if (CWMCU_write_i2c_block(sensor, CW_SET_INTERRUPT_ENABLE, &regValue, 1) < 0) {
            printk("--CWMCU-- Write CW_SET_INTERRUPT_ENABLE error.\n");
        } else {
            sensor->power_test_on = (regValue == 0);
            printk("CWMCU INT status=%d.\n", sensor->power_test_on);
        }
#endif
        break;


    default:
            break;
    }
    return count;
}

/* get calibrator data */
static ssize_t get_calibrator_data(struct device *dev, struct device_attribute *attr, char *buf)
{
    uint8_t data[10] = {0,};

    data[0] = sensor->calib_cmd;
    data[1] = sensor->calib_id;


    switch (sensor->calib_cmd) {

    case CWMCU_CALIBRATOR_SENSORS:
        printk("--CWMCU-- CWMCU_CALIBRATOR_STATUS\n");
        if (CWMCU_i2c_read(sensor, CW_CALIBRATOR_STATUS, &data[3], 1) >= 0) {
            printk("--CWMCU-- calibrator status = %d\n", data[3]);
            return sprintf(buf, "%d\n",(char)data[3]);
        } else {
            printk("--CWMCU-- calibrator => i2c error, calibrator status = %d\n", data[3]);
            return sprintf(buf, "-3\n");
        }
        break;

    case CWMCU_CALIBRATOR_GETBIAS:
        printk("--CWMCU-- CWMCU_CALIBRATOR_GETBIAS SensorID:%d\n",sensor->calib_id);

		if(sensor->calib_id == CW_ACCELERATION) {
		    if (CWMCU_i2c_read(sensor, CW_CALIBRATOR_GET_BIAS_ACC, &data[2], 6) <= 0) {
		        printk("--CWMCU-- read sensor[%d] calibrator fail[i2c]\n", sensor->calib_id);
		    }
		}
		else if(sensor->calib_id == CW_GYRO) {
		    if (CWMCU_i2c_read(sensor, CW_CALIBRATOR_GET_BIAS_GYRO, &data[2], 6) <= 0) {
		        printk("--CWMCU-- read sensor[%d] calibrator fail[i2c]\n", sensor->calib_id);
		    }
		}
		else if(sensor->calib_id == CW_PROXIMITY) {
		    if (CWMCU_i2c_read(sensor, CW_CALIBRATOR_GET_BIAS_PROXIMITY, &data[2], 8) <= 0) {
		        printk("--CWMCU-- read sensor[%d] calibrator fail[i2c]\n", sensor->calib_id);
		    }
		}
        break;
    }

    return sprintf(buf, "%d %d %d %d %d %d %d %d %d %d\n",
            data[0], data[1], data[2], data[3], data[4], data[5], data[6], data[7], data[8], data[9]);

}

static ssize_t set_calibrator_data(struct device *dev,struct device_attribute *attr,const char *buf, size_t count)
{
    int i = 0;
	int temp[10] = {0,};
    uint8_t data[10] = {0,};
	uint8_t calibData[10] = {0,};

	char source[512];
    char *pch;
    int buf_count=0;

    char *myBuf= source;

	strcpy(source,buf);
    while ((pch = strsep(&myBuf, ", ")) != NULL) {
        buf_count++;
    }

    /* calibrator command mode */
    if (buf_count == 2) {
        sscanf(buf, "%d %d",&temp[0], &temp[1]);
        sensor->calib_cmd = (uint8_t)temp[0];
        sensor->calib_id = (uint8_t)temp[1];
        printk("--CWMCU-- Calibrator=> cmd:%d id:%d\n", sensor->calib_cmd, sensor->calib_id);

		if(sensor->calib_cmd == CWMCU_CALIBRATOR_SENSORS)
			CWMCU_i2c_write(sensor, CW_CALIBRATOR_SENSOR, &sensor->calib_id, 1);
		else if(sensor->calib_cmd == CWMCU_CALIBRATOR_WRITEFLASH)
			CWMCU_i2c_write(sensor, CW_CALIBRATOR_FLASHWRITE_SENSOR_BIAS, &sensor->calib_id, 1);

    }
	/* write calibrator bias to mcu */
	else if (buf_count >= 6) {
        sscanf(buf, "%d %d %d %d %d %d %d %d %d %d\n",&temp[0],&temp[1],&temp[2],&temp[3],&temp[4],&temp[5],&temp[6],&temp[7],&temp[8],&temp[9]);
        for (i = 0; i < 10; i++) {
            data[i] = (uint8_t)temp[i];
        }
        sensor->calib_cmd = data[0];
        sensor->calib_id = data[1];
        printk("--CWMCU-- Calibrator=> cmd=%d id=%d setbias:%d %d %d %d %d %d %d %d\n", sensor->calib_cmd, sensor->calib_id,
				data[2],data[3],data[4],data[5],data[6],data[7],data[8],data[9]);

		calibData[0] = sensor->calib_id;
		memcpy(calibData+1,data+2,sizeof(uint8_t)*8);
		if(sensor->calib_cmd == CWMCU_CALIBRATOR_SETBIAS) {
			if(CWMCU_write_i2c_multiple(sensor, CW_CALIBRATOR_SET_BIAS, calibData, 9)<0) {
        		printk("--CWMCU-- set calibrator bias error[i2c].\n");
    		}
		}
        else {
			printk("--CWMCU-- set calibrator bias parameter errorr.\n");
		}

    } else {
        printk("--CWMCU-- input parameter incorrect !!! | %zd\n",count);
        return count;
    }

    return count;
}

static ssize_t selftest_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	uint8_t data[2] = {0,};

	printk("--CWMCU-- selftest show\n");

	if (CWMCU_i2c_read(sensor, CW_SELFTEST_STATUS, &data[0], 1) <= 0) {
		printk("--CWMCU-- get selftest status error[i2c]\n");
		return sprintf(buf, "-3\n");
	}

	return sprintf(buf, "%d\n",(char)data[0]);
}
static ssize_t selftest_set(struct device *dev,struct device_attribute *attr,const char *buf, size_t count)
{
	int temp[2] = {0,};

	sscanf(buf, "%d",&temp[0]);
    sensor->selftest_id = (uint8_t)temp[0];
	printk("--CWMCU-- selftest=> id:%d\n", sensor->selftest_id);

	if (CWMCU_i2c_write(sensor, CW_SELFTEST_SENSOR, &sensor->selftest_id, 1) < 0) {
		printk("--CWMCU-- Sensor:%d selftest error[i2c]\n",sensor->selftest_id);
	}

	return count;
}

static ssize_t version_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    uint8_t data[6] = {0,};

    printk("--CWMCU-- %s\n", __func__);

    if (cwmcu_get_version())
        sensor->mcu_status = CW_INVAILD;
    else {
        sensor->mcu_status = CW_VAILD;

        if (CWMCU_i2c_read(sensor, CW_FWVERSION, data, 2) >= 0) {
            printk("--CWMCU-- CHECK_FIRMWARE_VERSION : %d.%d\n", data[0],data[1]);
        }
    }

    return snprintf(buf, 8, "%d.%d\n", data[0],data[1]);
}

static ssize_t timestamp_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    uint8_t data[4] = {0,};
	uint32_t timestamp = 0;

    printk("--CWMCU-- %s\n", __func__);

	if (sensor->mcu_mode == CW_NORMAL && sensor->mcu_power == CW_NORMAL) {
        if (CWMCU_i2c_read(sensor, CW_MCU_TIMESTAMP, data, 4) >= 0) {
			timestamp = (uint32_t)(data[3]<<24) | (uint32_t)(data[2]<<16) | (uint32_t)(data[1]<<8) | (uint32_t)data[0];
			printk("--CWMCU-- mcu timestamp:%u\n", timestamp);
		}
    }
	
    return snprintf(buf, 32, "%u\n", timestamp);
}

static ssize_t firmware_name_set(struct device *dev,struct device_attribute *attr,const char *buf, size_t count)
{
    printk("--CWMCU-- %s\n", __func__);
    sscanf(buf, "%s\n", sensor->fw_name);

    printk("--CWMCU-- sensor->fw_path : %s\n", sensor->fw_name);

    return count;
}

static ssize_t firmware_name_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("--CWMCU-- %s : %s\n", __func__, sensor->fw_name);

    return snprintf(buf, 50, "%s\n", sensor->fw_name);
}


static ssize_t verbose_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("--CWMCU-- %s\n", __func__);

    return snprintf(buf, 50, "%d\n", flagVerbose);
}

static ssize_t verbose_set(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    sscanf(buf, "%d\n", &flagVerbose);
    printk("--CWMCU-- %s, parsed %d\n", __func__, flagVerbose);

    return count;
}

static ssize_t led_control_set(struct device *dev,struct device_attribute *attr,const char *buf, size_t count)
{
    int value;
    uint8_t data;

    sscanf(buf, "%d\n", &value);
    data = (uint8_t)value;
    printk("--CWMCU-- %s led value is %d\n", __func__, data);
    if (sensor->mcu_mode == CW_NORMAL && sensor->mcu_power == CW_NORMAL) {
        CWMCU_i2c_write(sensor, CW_LED_CTRL, &data, 1);
        if (CWMCU_i2c_write(sensor, CW_LED_CTRL, &data, 1) < 0)
            printk("led_control_set fail and CWMCU_i2c_write error\n");
    }
    return count;
}

static ssize_t power_control_set(struct device *dev,struct device_attribute *attr,const char *buf, size_t count)
{
    int value;
    uint8_t data;

    printk("--CWMCU-- %s\n", __func__);
    sscanf(buf, "%d\n", &value);
    data = (uint8_t)value;

    if(data==0xFF)
        sensor->mcu_power = CW_NORMAL;
    else
        sensor->mcu_power = CW_BOOT;

    CWMCU_i2c_write(sensor, CW_POWER_CTRL, &data, 1);

    return count;
}

#define MAG_VENDOR_AKM           1
#define MAG_VENDOR_ALPS          2

static uint8_t mag_vendor = MAG_VENDOR_ALPS;
static void get_mag_vendor(void)
{
    printk("--CWMCU-- %s\n", __func__);

    if (CWMCU_i2c_read(sensor, CW_MAG_GET_VENDOR_TYPE, &mag_vendor, 1) >= 0) {
        printk("--CWMCU-- get mag vendor: %d\n", mag_vendor);

        if ((mag_vendor != MAG_VENDOR_AKM) && (mag_vendor != MAG_VENDOR_ALPS)) {
            mag_vendor = MAG_VENDOR_ALPS;
            printk("--CWMCU-- mag vendor error, the default vendor is: %d\n", mag_vendor);
        }
    } else {
        mag_vendor = MAG_VENDOR_ALPS;
        printk("--CWMCU-- cannot get mag vendor, the default vendor is: %d\n", mag_vendor);
    }
}

static ssize_t get_magnetic_vendor(struct device *dev, struct device_attribute *attr,  char *buf)
{
    printk("--CWMCU-- %s\n", __func__);
    printk("--CWMCU-- mag vendor (local): %d\n", mag_vendor);

    return snprintf(buf, 8, "%d\n", mag_vendor);
}

static ssize_t psensor_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    uint8_t data = 0;

    printk("--CWMCU-- %s\n", __func__);

    if (CWMCU_i2c_read(sensor, CW_PSENSOR_RAW_DATA_GET, &data, 1) >= 0) {
        printk("Get P-sensor RAWDATA: %d\n", data);
    }

    return snprintf(buf, 8, "%d\n", data);
}



static char sensor_type_desc[CW_SENSORS_ID_END][31] = {
"acceleration",
"magnetic",
"gyro",
"light",
"proximity",
"pressure",
"heartbeat",
"orientation",
"rotationvector",
"linearacceleration",
"gravity",
"magnetic_uncalibrated",
"gyroscope_uncalibrated",
"game_rotation_vector",
"geomagnetic_rotation_vector",
"step_detector",
"step_counter",
"significant_motion",
"tilt",
"pdr",
"single_snap",
"double_snap",
"tap",
"context_aware",
"rear_camera",
"proximity_gesture",
"light_rgb",
};

	
static ssize_t sns_info_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    int len = 0;
    //uint8_t id;
    uint8_t data[8];
    uint32_t mcu_enable_list[CW_HANDLE_ID_END] = {0,};
	
	int id,hd;
	uint8_t mHandle, mSensorId;
    uint8_t rData[2],wData[2];
    uint8_t sensor_report_rate[CW_HANDLE_ID_END][CW_SENSORS_ID_END];
	
	printk("--CWMCU-- %s\n", __func__);
	
	memset(sensor_report_rate,0,sizeof(uint8_t)*CW_SENSORS_ID_END*CW_HANDLE_ID_END);
	for(hd=0;hd<CW_HANDLE_ID_END;hd++)
	{
		mHandle = (uint8_t)hd;
		wData[1] = mHandle;
		for(id=0;id<CW_SENSORS_ID_END;id++)
		{
			mSensorId = (uint8_t)id;
			wData[0] = mSensorId;
			if(CWMCU_write_i2c_block(sensor, CW_SENSOR_DELAY_ID_SET, wData, 2) >= 0) {
				if (CWMCU_i2c_read(sensor, CW_SENSOR_DELAY_GET, rData, 1) >= 0) {
					sensor_report_rate[hd][id] = rData[0];
					printk("--CWMCU-- sensor report rate(ms)  sensor[%d][%d] => %d\n",hd,id, sensor_report_rate[hd][id]);
				} else {
					printk("--CWMCU-- Get sensor report rate error. sensorId=%d handle=%d\n",mSensorId,mHandle);
				}
			} else {
				printk("--CWMCU-- Set sensor ID error when get report rate. sensorId=%d handle=%d\n",mSensorId,mHandle);
			}
		}
	}
   
    if (CWMCU_i2c_read(sensor, CW_ENABLE_STATUS, data, 8) < 0) {
        printk("--CWMCU-- get mcu enable list error.\n");
    }
    mcu_enable_list[CW_NONWAKEUP] = (uint32_t)data[3]<< 24 | (uint32_t)data[2]<< 16 | (uint32_t)data[1]<< 8 | data[0];
	mcu_enable_list[CW_WAKEUP] = (uint32_t)data[7]<< 24 | (uint32_t)data[6]<< 16 | (uint32_t)data[5]<< 8 | data[4];

    len += sprintf(buf + len,
                        "mcu_mode                  : %d\n" \
                        "mcu_status                : %d\n" \
                        "mcu_power                 : %d\n" \
                        "mcu_screen                : %d\n" \
                        "mcu_reset                 : %d\n" \
                        "enabled_list_nonwp        : 0x%x\n" \
						"enabled_list_wp           : 0x%x\n" \
                        "mcu_enabled_list_nonwp    : 0x%x\n" \
						"mcu_enabled_list_wp       : 0x%x\n" \
                        "batched_list_nonwp        : 0x%x\n" \
						"batched_list_wp           : 0x%x\n" \
                        "update_list_nonwp         : 0x%x\n" \
						"update_list_wp            : 0x%x\n" \
                        "batch_timeout_nonwp       : %u\n" \
						"batch_timeout_wp          : %u\n" \
                        "interrupt_status          : 0x%x\n" \
                        "power_on_list             : %d\n", \
                        sensor->mcu_mode,
                        sensor->mcu_status,
                        sensor->mcu_power,
                        sensor->mcu_screen,
                        sensor->mcu_reset,
                        sensor->enabled_list[CW_NONWAKEUP],
						sensor->enabled_list[CW_WAKEUP],
                        mcu_enable_list[CW_NONWAKEUP],
						mcu_enable_list[CW_WAKEUP],
                        sensor->batched_list[CW_NONWAKEUP],
						sensor->batched_list[CW_WAKEUP],
                        sensor->update_list[CW_NONWAKEUP],
						sensor->update_list[CW_WAKEUP],
//                        sensor->suspend_enabled_list,
//                        SUSPEND_ENABLE_LIST,
                        sensor->batch_timeout[CW_NONWAKEUP],
						sensor->batch_timeout[CW_WAKEUP],
                        sensor->interrupt_status,
                        sensor->power_on_list);

    len += sprintf(buf + len, "\n.......................................................................................................\n");
    len += sprintf(buf + len, "description                   id :  handle  active  batch  update  report(mcu)  report       sensor_batch\n");
    len += sprintf(buf + len, "                                 :                                 period(ms)   period(ms)   timeout(ms)\n");
    for (id = 0; id < CW_SENSORS_ID_END; id++) {
        len += sprintf(buf + len, "%-30s%2d :  non-wakeup %6s  %5s  %6s  %7d      %7d           %7d\n",
                sensor_type_desc[id],
                id,
                (sensor->enabled_list[CW_NONWAKEUP] & (1 << id)) ? "Y" : "",
                (sensor->batched_list[CW_NONWAKEUP] & (1 << id)) ? "Y" : "",
                (sensor->update_list[CW_NONWAKEUP] & (1 << id)) ? "Y" : "",
				sensor_report_rate[CW_NONWAKEUP][id],
                sensor->report_period[CW_NONWAKEUP][id],
                sensor->sensor_timeout[CW_NONWAKEUP][id]);
    }
	/*
	len += sprintf(buf + len, "\n.......................................................................................................\n");
    len += sprintf(buf + len, "description                   id :  handle  active  batch  update  report       sensor_batch\n");
    len += sprintf(buf + len, "                                 :                                 period(ms)   timeout(ms)\n");
    for (id = 0; id < CW_SENSORS_ID_END; id++) {
        len += sprintf(buf + len, "%-30s%2d :  wakeup %6s  %5s  %6s  %7d      %7d           %7d\n",
                sensor_type_desc[id],
                id,
                (sensor->enabled_list[CW_WAKEUP] & (1 << id)) ? "Y" : "",
                (sensor->batched_list[CW_WAKEUP] & (1 << id)) ? "Y" : "",
                (sensor->update_list[CW_WAKEUP] & (1 << id)) ? "Y" : "",
				sensor_report_rate[CW_WAKEUP][id],
                sensor->report_period[CW_WAKEUP][id],
                sensor->sensor_timeout[CW_WAKEUP][id]);
    }
	*/

    return len;
}


static DEVICE_ATTR(enable, 0644, active_show, active_set);
static DEVICE_ATTR(delay_ms, 0644, interval_show, interval_set);
static DEVICE_ATTR(batch, 0644, batch_show, batch_set);
static DEVICE_ATTR(flush, 0644, flush_show, flush_set);

//static DEVICE_ATTR(firmware_update_i2c, 0644, get_firmware_update_i2, set_firmware_update_i2);
static DEVICE_ATTR(firmware_update_cmd, 0644, firmware_update_cmd_show, firmware_update_cmd_set);
static DEVICE_ATTR(firmware_name, 0644, firmware_name_show, firmware_name_set);
static DEVICE_ATTR(mcu_cmd, 0644, NULL, set_mcu_cmd);
static DEVICE_ATTR(calibrator_cmd, 0644, get_calibrator_data, set_calibrator_data);
static DEVICE_ATTR(selftest, 0644, selftest_show, selftest_set);
static DEVICE_ATTR(timestamp, 0644, timestamp_show, NULL);
static DEVICE_ATTR(version, 0644, version_show, NULL);
static DEVICE_ATTR(verbose, 0644, verbose_show, verbose_set);
static DEVICE_ATTR(led_control, 0644, NULL, led_control_set);
static DEVICE_ATTR(power_control, 0644, NULL, power_control_set);
static DEVICE_ATTR(mag_vendor, 0644, get_magnetic_vendor, NULL);
static DEVICE_ATTR(psensor, 0644, psensor_show, NULL);
static DEVICE_ATTR(sns_info, 0644, sns_info_show, NULL);

static struct attribute *sysfs_attributes[] = {
	&dev_attr_enable.attr,
	&dev_attr_delay_ms.attr,
	&dev_attr_batch.attr,
	&dev_attr_flush.attr,
	//&dev_attr_firmware_update_i2c.attr,
	&dev_attr_firmware_update_cmd.attr,
    &dev_attr_firmware_name.attr,
	&dev_attr_mcu_cmd.attr,
	&dev_attr_calibrator_cmd.attr,
	&dev_attr_selftest.attr,
	&dev_attr_timestamp.attr,
	&dev_attr_version.attr,
	&dev_attr_verbose.attr,
	&dev_attr_led_control.attr,
	&dev_attr_power_control.attr,
	&dev_attr_mag_vendor.attr,
	&dev_attr_psensor.attr,
    &dev_attr_sns_info.attr,
	NULL
};

static struct attribute_group sysfs_attribute_group = {
	.attrs = sysfs_attributes
};

/*=======input device==========*/

static int CWMCU_init_input_device(struct CWMCU_data *sensor)
{
	int err = 0;
	
	sensor->input = input_allocate_device();
    if (!sensor->input) {
        printk("Failed to allocate CwMcuSensor input device\n");
        err = -ENOMEM;
		return err;
    }
    sensor->input->name = CWMCU_I2C_NAME;
    sensor->input->id.bustype = BUS_I2C;
    sensor->input->dev.parent = &sensor->client->dev;	

    set_bit(EV_ABS, sensor->input->evbit);
    input_set_abs_params(sensor->input, CW_ABS_X, -DPS_MAX, DPS_MAX, 0, 0);
    input_set_abs_params(sensor->input, CW_ABS_Y, -DPS_MAX, DPS_MAX, 0, 0);
    input_set_abs_params(sensor->input, CW_ABS_Z, -DPS_MAX, DPS_MAX, 0, 0);
    input_set_abs_params(sensor->input, CW_ABS_X1, -DPS_MAX, DPS_MAX, 0, 0);
    input_set_abs_params(sensor->input, CW_ABS_Y1, -DPS_MAX, DPS_MAX, 0, 0);
    input_set_abs_params(sensor->input, CW_ABS_Z1, -DPS_MAX, DPS_MAX, 0, 0);
    input_set_abs_params(sensor->input, CW_ABS_TIMEDIFF, -DPS_MAX, DPS_MAX, 0, 0);
    input_set_abs_params(sensor->input, CW_ABS_TIMESTAMP, -DPS_MAX, DPS_MAX, 0, 0);
    input_set_abs_params(sensor->input, CW_ABS_TIMESYNC, -DPS_MAX, DPS_MAX, 0, 0);
    input_set_abs_params(sensor->input, CW_ABS_MAG_ACCURACY, -DPS_MAX, DPS_MAX, 0, 0);
	input_set_abs_params(sensor->input, CW_ABS_ORI_ACCURACY, -DPS_MAX, DPS_MAX, 0, 0);

	err = input_register_device(sensor->input);
    if (err) {
        pr_err("Failed to register CwMcuSensor input device\n");
        return err;
    }

	return 0;
}


static int CWMCU_suspend(struct device *dev)
{
	u8 regVaule = 1;
	printk(KERN_DEBUG "--CWMCU--%s\n", __func__);
#ifdef CONFIG_SNS_HUB_DEBUG
    if (sensor->power_test_on) {
        sensor->is_suspend = true;
        printk("CWMCU: Power testing, do nothing.\n");
        return 0;
    }
#endif
	if(CWMCU_i2c_write(sensor, CW_SYSTEM_SUSPEND, &regVaule, 1) < 0) {
		printk("--CWMCU-- Set system suspend error[i2c]\n");
	}
    sensor->is_suspend = true; // Disable all following i2c ops in IRQ handler until resume.
    printk("CWMCU waitting for queued works done\n");
    flush_work(&sensor->work);
    printk("CWMCU suspend completed\n");
    if (wake_lock_active(&wakeup_event_wake_lock)) {
        printk("CWMCU suspend when wakelock activated, return fail\n");
        return -1;
    } else
        return 0;
}

static int CWMCU_resume(struct device *dev)
{
	u8 regVaule = 0;
	printk(KERN_DEBUG "--CWMCU--%s\n", __func__);
#ifdef CONFIG_SNS_HUB_DEBUG
    if (sensor->power_test_on) {
        sensor->is_suspend = false;
        printk("CWMCU: Power testing, do nothing.\n");
        return 0;
    }
#endif
    usleep(10000);
    sensor->is_suspend = false;

	if(CWMCU_i2c_write(sensor, CW_SYSTEM_RESUME, &regVaule, 1) < 0) {
		printk("--CWMCU-- Set system resume error[i2c]\n");
	}
    printk("CWMCU resume completed\n");
	return 0;
}

#ifdef CWMCU_INTERRUPT
static irqreturn_t CWMCU_interrupt_thread(int irq, void *data)
{
	if(sensor->sensorhub_data_mode & 2) {
        //	printk(KERN_DEBUG "--CWMCU--%s in\n", __func__); //zhujp2 modify 2015.01.29
        VERBOSE("--CWMCU--%s in\n", __func__); // zhujp2 modify 2015.01.29
		if ((sensor->mcu_mode == CW_BOOT) || sensor->mcu_status == CW_INVAILD) {
			printk(KERN_DEBUG "--CWMCU--%s sensor->mcu_mode = %d\n", __func__, sensor->mcu_mode);
			return IRQ_HANDLED;
		}
        if (!sensor->is_suspend) {
            schedule_work(&sensor->work);
        }
	}

	return IRQ_HANDLED;
}

/*  function for interrupt work  */
static void cwmcu_work_report(struct work_struct *work)
{
	uint8_t temp[6] = {0,};
    //char mStatus;

    //	printk("--CWMCU--%s in\n", __func__);  //zhujp2 modify 2015.01.29
    VERBOSE("--CWMCU--%s in\n", __func__); //zhujp2 modify 2015.01.29

    if ((sensor->mcu_mode == CW_BOOT) || sensor->mcu_status == CW_INVAILD) {
        printk("--CWMCU--%s sensor->mcu_mode = %d\n", __func__, sensor->mcu_mode);
        return;
    }

    /* check mcu interrupt status */
    if (CWMCU_i2c_read(sensor, CW_INTERRUPT_STATUS, temp, 2) >= 0) {
        sensor->interrupt_status = (u32)temp[1] << 8 | (u32)temp[0];
    //    printk("--CWMCU-- sensor->interrupt_status~ = 0x%2X\n", sensor->interrupt_status);        // zhujp2  modify 2015.01.29
    VERBOSE("--CWMCU-- sensor->interrupt_status~ = 0x%2X\n", sensor->interrupt_status);  //zhujp2  modify  2015.01.29
    } else {
        printk("--CWMCU-- check interrupt_status failed~!!\n");
        sensor->interrupt_status = 0;
    }


    /* MCU reset */
    if (sensor->interrupt_status & (1<<INTERRUPT_MCU_INIT)) {
		if(sensor->mcu_reset > 0) {
			cwmcu_restore_status();
			printk( "--CWMCU-- Check MCU restore status.\n");
		}
		sensor->mcu_reset++;
    }

	/* MCU time sync */
    if (sensor->interrupt_status & (1<<INTERRUPT_TIME_SYNC)) {
		cwmcu_send_time_sync_event();
		printk( "--CWMCU-- MCU Time sync event send.\n");
    }



	//cwmcu_powermode_switch(SWITCH_POWER_INTERRUPT, 1);

#if 1
	/* read MCU error log data */
    if (sensor->interrupt_status & (1<<INTERRUPT_ERROR_LOG)) {
        cwmcu_error_log_read(sensor);
    }

    /* read MCU debug log data */
    if (sensor->interrupt_status & (1<<INTERRUPT_DEBUG_LOG)) {
        cwmcu_debug_log_read(sensor);
    }
#endif

	/* read sensor data of normal mode*/
    if (sensor->interrupt_status & (1<<INTERRUPT_DATAREADY)) {
        CWMCU_read(sensor);
    }

	/* read sensor data of stream nonwakeup mode*/
    if (sensor->interrupt_status & (1<<INTERRUPT_STREAM_NONWAKEUP)) {
        cwmcu_stream_mode_read(sensor,CW_NONWAKEUP);
    }
	/* read sensor data of stream wakeup mode*/
    if (sensor->interrupt_status & (1<<INTERRUPT_STREAM_WAKEUP)) {
        cwmcu_stream_mode_read(sensor,CW_WAKEUP);
    }
	
	/* read sensor data of batch nonwakeup mode*/
    if (sensor->interrupt_status & (1<<INTERRUPT_BATCH_NONWAKEUP)) {
        cwmcu_batch_mode_read(sensor,CW_NONWAKEUP);
    }
	/* read sensor data of batch wakeup mode*/
    if (sensor->interrupt_status & (1<<INTERRUPT_BATCH_WAKEUP)) {
        cwmcu_batch_mode_read(sensor,CW_WAKEUP);
    }
	/* batch flush */
    if (sensor->interrupt_status & (1<<INTERRUPT_BATCH_FLUSH)) {
		cwmcu_batch_mode_read(sensor,CW_NONWAKEUP);
        cwmcu_batch_mode_read(sensor,CW_WAKEUP);
    }

    VERBOSE("--CWMCU--%s out\n", __func__); 
}
#endif

static int cwstm_parse_dt(struct device *dev, struct CWMCU_data *sensor)
{
	struct device_node *np = dev->of_node;
	int ret = 0;

	ret = of_get_named_gpio(np, "cwstm,irq-gpio", 0);
	if (ret < 0) {
		pr_err("failed to get \"cwstm,irq_gpio\"\n");
		goto err;
	}
	sensor->irq_gpio = ret;

    ret = of_get_named_gpio(np, "cwstm,mic-gpio", 0);
    if (ret < 0) {
        pr_err("failed to get \"cwstm,mic-gpio\"\n");
        goto err;
    }
    sensor->mic_gpio = ret;

	ret = of_get_named_gpio(np, "cwstm,reset-gpio", 0);
	if (ret < 0) {
		pr_err("failed to get \"reset\"\n");
		goto err;
	}
	sensor->reset_gpio = ret;

#ifdef CONFIG_SNR_HUB_DVT1_COMPATIBLE
	ret = of_get_named_gpio(np, "cwstm,reset-gpio-dvt1", 0);
	if (ret < 0) {
		pr_err("failed to get \"reset-dvt1\"\n");
		goto err;
	}
	sensor->reset_gpio_dvt1 = ret;
#endif

	ret = of_get_named_gpio(np, "cwstm,wakeup-gpio", 0);
	if (ret < 0) {
		pr_err("failed to get \"wakeup\"\n");
		goto err;
	}
	sensor->wakeup_gpio = ret;

	ret = of_get_named_gpio(np, "cwstm,compass-reset-gpio", 0);
	if (ret < 0) {
		pr_err("failed to get \"reset\"\n");
		goto err;
	}
	sensor->compass_reset_gpio = ret;

err:
	return ret;
}

static int cwstm_power_on(bool on)
{
	int rc = 0;

	if (!on)
		goto power_off;

	rc = regulator_enable(sensor->vdd);
	if (rc) {
		dev_err(&sensor->client->dev,
			"Regulator vdd enable failed rc=%d\n", rc);
		return rc;
	}

	rc = regulator_enable(sensor->vcc_i2c);
	if (rc) {
		dev_err(&sensor->client->dev,
			"Regulator vcc_i2c enable failed rc=%d\n", rc);
		regulator_disable(sensor->vdd);
	}

	return rc;

power_off:
	rc = regulator_disable(sensor->vdd);
	if (rc) {
		dev_err(&sensor->client->dev,
			"Regulator vdd disable failed rc=%d\n", rc);
		return rc;
	}

	rc = regulator_disable(sensor->vcc_i2c);
	if (rc) {
		dev_err(&sensor->client->dev,
			"Regulator vcc_i2c disable failed rc=%d\n", rc);
		rc = regulator_enable(sensor->vdd);
		return rc;
	}

	return rc;
}

static int cwstm_power_init(bool on)
{
	int rc;

	if (!on)
		goto pwr_deinit;

	sensor->vdd = regulator_get(&sensor->client->dev, "vdd");
	if (IS_ERR(sensor->vdd)) {
		rc = PTR_ERR(sensor->vdd);
		dev_err(&sensor->client->dev,
			"Regulator get failed vdd rc=%d\n", rc);
		return rc;
	}

	if (regulator_count_voltages(sensor->vdd) > 0) {
		rc = regulator_set_voltage(sensor->vdd, FT_VTG_MIN_UV,
					   FT_VTG_MAX_UV);
		if (rc) {
			dev_err(&sensor->client->dev,
				"Regulator set_vtg failed vdd rc=%d\n", rc);
			goto reg_vdd_put;
		}
	}

	sensor->vcc_i2c = regulator_get(&sensor->client->dev, "vcc_i2c");
	if (IS_ERR(sensor->vcc_i2c)) {
		rc = PTR_ERR(sensor->vcc_i2c);
		dev_err(&sensor->client->dev,
			"Regulator get failed vcc_i2c rc=%d\n", rc);
		goto reg_vdd_set_vtg;
	}

	if (regulator_count_voltages(sensor->vcc_i2c) > 0) {
		rc = regulator_set_voltage(sensor->vcc_i2c, FT_I2C_VTG_MIN_UV,
					   FT_I2C_VTG_MAX_UV);
		if (rc) {
			dev_err(&sensor->client->dev,
			"Regulator set_vtg failed vcc_i2c rc=%d\n", rc);
			goto reg_vcc_i2c_put;
		}
	}

	return 0;

reg_vcc_i2c_put:
	regulator_put(sensor->vcc_i2c);
reg_vdd_set_vtg:
	if (regulator_count_voltages(sensor->vdd) > 0)
		regulator_set_voltage(sensor->vdd, 0, FT_VTG_MAX_UV);
reg_vdd_put:
	regulator_put(sensor->vdd);
	return rc;

pwr_deinit:
	if (regulator_count_voltages(sensor->vdd) > 0)
		regulator_set_voltage(sensor->vdd, 0, FT_VTG_MAX_UV);

	regulator_put(sensor->vdd);

	if (regulator_count_voltages(sensor->vcc_i2c) > 0)
		regulator_set_voltage(sensor->vcc_i2c, 0, FT_I2C_VTG_MAX_UV);

	regulator_put(sensor->vcc_i2c);
	return 0;
}

static int CWMCU_i2c_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	int error;
	int h = 0, i = 0;

    printk(KERN_DEBUG "--CWMCU-- %s\n", __func__);

    printk(KERN_DEBUG "CWMCU SW CFG: "
#ifdef CWMCU_INTERRUPT
    "INT/" 
#else
    "POLL/"
#endif
#ifdef CONFIG_SNR_HUB_DVT1_COMPATIBLE
    "DVT1_OK/" 
#endif
#ifdef CONFIG_SNS_HUB_DEBUG
    "DEBUG/"  
#endif
#ifdef CONFIG_SNR_HUB_LG_HACK
    "LG_HACK/"
#endif
#ifdef CONFIG_SNR_HUB_MAG_LEAK_HACK
    "MAG_LEAK_HACK/"
#endif
#ifdef CONFIG_SNR_HUB_I2C_HACK
    "I2C_HACK/"
#endif
    "\n");

	dev_dbg(&client->dev, "%s:\n", __func__);

	if (!i2c_check_functionality(client->adapter, I2C_FUNC_I2C)) {
		dev_err(&client->dev, "--CWMCU-- i2c_check_functionality error\n");
		return -EIO;
	}

	sensor = kzalloc(sizeof(struct CWMCU_data), GFP_KERNEL);
	if (!sensor) {
		printk(KERN_DEBUG "--CWMCU-- kzalloc error\n");
		return -ENOMEM;
	}

	sensor->client = client;
	i2c_set_clientdata(client, sensor);

	error = cwstm_parse_dt(&client->dev, sensor);
	if (error < 0) {
		pr_err("failed to parse device tree: %d\n", error);
		goto err_parse_dt;
	}
	printk("%s: irq = %d, wakeup = %d, reset = %d\n", __func__, sensor->irq_gpio, sensor->wakeup_gpio, sensor->reset_gpio);
#ifdef CONFIG_SNR_HUB_DVT1_COMPATIBLE
	printk("%s: reset-dvt1 = %d\n", __func__, sensor->reset_gpio_dvt1);
#endif
	gpio_request(sensor->wakeup_gpio, "cwstm,wakeup-gpio");
	gpio_direction_output(sensor->wakeup_gpio, 1);
	gpio_request(sensor->reset_gpio, "cwstm,reset-gpio");
	gpio_request(sensor->mic_gpio, "cwstm,mic-gpio");
#ifdef CONFIG_SNR_HUB_DVT1_COMPATIBLE
	gpio_request(sensor->reset_gpio_dvt1, "cwstm,reset-gpio-dvt1");
#endif
	gpio_request(sensor->irq_gpio, "cwstm,irq-gpio");
	gpio_request(sensor->compass_reset_gpio, "cwstm,compass-reset-gpio");

#ifdef CONFIG_SNR_HUB_MAG_LEAK_HACK
	gpio_direction_output(sensor->compass_reset_gpio, 0);
	udelay(5);
	gpio_set_value(sensor->compass_reset_gpio, 1);
	udelay(100);
#else
	gpio_direction_output(sensor->compass_reset_gpio, 1);
#endif

	error = cwstm_power_init(true);
	if (error) {
		dev_err(&client->dev, "power init failed");
	}

	error = cwstm_power_on(true);
	if (error) {
		dev_err(&client->dev, "power on failed");
	}

	CWMCU_reset_mcu(true); // Assure that we reset MCU during driver probe.
	
	error = CWMCU_init_input_device(sensor);
	if (error)
		goto exit_free_input;

	
	error = sysfs_create_group(&sensor->input->dev.kobj,
					&sysfs_attribute_group);
	if (error)
		goto err_free_mem;

	for (h = 0; h < CW_HANDLE_ID_END; h++) {
		for (i = 0; i < CW_SENSORS_ID_END; i++) {
			sensor->report_period[h][i] = 200;
			sensor->time_diff[h][i] = 0;
			sensor->pre_sensors_time[h][i] = 0;
		}
	}
	
	
	sensor->mcu_mode = CW_NORMAL;
	sensor->sensorhub_data_mode = 2;

#ifdef CWMCU_INTERRUPT
	sensor->client->irq = gpio_to_irq(sensor->irq_gpio);

	printk( "--CWMCU--client->irq  =%d, irq_gpio %d~!!\n", sensor->client->irq, sensor->irq_gpio);
    //gpio_request(sensor->irq_gpio, "cwstm,irq-gpio");
    gpio_direction_output(sensor->irq_gpio, 1);
    usleep(20000);
    pr_info("%s:irq gpio vlaue %d\n", __func__, gpio_get_value(sensor->irq_gpio));
    gpio_direction_input(sensor->irq_gpio);
    usleep(5000);

    wake_lock_init(&fwupdate_wake_lock, WAKE_LOCK_SUSPEND, "cwmcu_fwupdate_wake_lock");
    wake_lock_init(&wakeup_event_wake_lock, WAKE_LOCK_SUSPEND, "cwmcu_events_wake_lock");

	if (sensor->client->irq > 0) {

		error = request_threaded_irq(sensor->client->irq, NULL,
						   CWMCU_interrupt_thread,
						   IRQF_TRIGGER_FALLING | IRQF_ONESHOT,
						   "cwmcu", sensor);
		if (error < 0) {
				pr_err("request irq %d failed\n", sensor->client->irq);
				goto exit_destroy_mutex;
		}
		disable_irq(client->irq);
		INIT_WORK(&sensor->work, cwmcu_work_report);
		enable_irq(client->irq);
		enable_irq_wake(sensor->client->irq);
	}
#endif

    //i2c_set_clientdata(client, sensor);
    usleep(10000);

    if (cwmcu_get_version())
        sensor->mcu_status = CW_INVAILD;
    else {
        sensor->mcu_status = CW_VAILD;
        get_mag_vendor();
    }

#ifdef CONFIG_SNS_HUB_DEBUG
    sensor->power_test_on = false;
#endif
    sensor->is_suspend = false;
	
	printk(KERN_DEBUG "--CWMCU-- CWMCU_i2c_probe success!\n");

	return 0;

exit_free_input:
	input_free_device(sensor->input);
err_free_mem:
err_parse_dt:
#ifdef CWMCU_INTERRUPT
exit_destroy_mutex:
	free_irq(sensor->irq_gpio, sensor);
#endif	
	kfree(sensor);
	return error;
}

static int CWMCU_i2c_remove(struct i2c_client *client)
{
    struct CWMCU_data *sensor = i2c_get_clientdata(client);
    wake_lock_destroy(&fwupdate_wake_lock);
    wake_lock_destroy(&wakeup_event_wake_lock);
    kfree(sensor);
    return 0;
}

static struct of_device_id cwstm_match_table[] = {
	{ .compatible = "cwstm,cwstm32",},
	{ },
};

static const struct dev_pm_ops CWMCU_pm_ops = {
	.suspend = CWMCU_suspend,
	.resume = CWMCU_resume
};

static const struct i2c_device_id CWMCU_id[] = {
	{ CWMCU_I2C_NAME, 0 },
	{ }
};

MODULE_DEVICE_TABLE(i2c, CWMCU_id);

static struct i2c_driver CWMCU_driver = {
	.driver = {
		.name = CWMCU_I2C_NAME,
		.owner = THIS_MODULE,
		.pm = &CWMCU_pm_ops,
		.of_match_table = cwstm_match_table,
	},
	.probe    = CWMCU_i2c_probe,
	.remove   = CWMCU_i2c_remove,
	.id_table = CWMCU_id,
};

static int __init CWMCU_i2c_init(void){
	printk(KERN_DEBUG "CWMCU_i2c_init\n");
	return i2c_add_driver(&CWMCU_driver);
}

static void __exit CWMCU_i2c_exit(void){
	i2c_del_driver(&CWMCU_driver);
}

module_init(CWMCU_i2c_init);
module_exit(CWMCU_i2c_exit);

MODULE_DESCRIPTION("CWMCU I2C Bus Driver");
MODULE_AUTHOR("CyWee Group Ltd.");
MODULE_LICENSE("GPL");
